function [StateVar,ShockVar,MeasureVar,ParameterVar] = SummarizeDSGEPredictions(StateCov,ShockCov,MeasureCov,ParameterCov)
% SummarizeDSGEPredictions: Summarizes the information about the prediction uncertainty in the DSGE model.
%
% USAGE:
%
%       [StateVar,ShockVar,MeasureVar,ParameterVar] = SummarizeDSGEPredictions(StateCov,ShockCov,MeasureCov,ParameterCov)
%
% REQUIRED INPUT:  StateCov (n x n x h) matrix with the estimated state uncertainty.
%
%                  ShockCov (n x n x h) matrix with the estimated shock uncertainty.
%
%                  MeasureCov (n x n) matrix the the estimated measurement error uncertainty.
%
%                  ParameterCov (n x n x h) matrix with the estimated parameter uncertainty.
%
% REQUIRED OUTPUT: StateVar (nxh) matrix with the share of the overvall uncertainty for the n observed
%                           variables and the h predictions that is due to state uncertainty.
%
%                  ShockVar (nxh) matrix with the share of the overvall uncertainty for the n observed
%                           variables and the h predictions that is due to shock uncertainty.
%
%                  MeasureVar (nxh) matrix with the share of the overvall uncertainty for the n observed
%                           variables and the h predictions that is due to measurement errors.
%
%                  ParameterVar (nxh) matrix with the share of the overvall uncertainty for the n observed
%                           variables and the h predictions that is due to parameter uncertainty.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: February 23, 2007.
%                        This version: January 22, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 12-03-2007: Checks to make sure that all relevant elements are non-negative.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize the output
%
n = size(StateCov,1);
h = size(StateCov,3);
StateVar = zeros(n,h);
ShockVar = zeros(n,h);
MeasureVar = zeros(n,h);
ParameterVar = zeros(n,h);
%
% keep just the variances
%
for i=1:h;
   StateVar(:,i) = diag(StateCov(:,:,i));
   ShockVar(:,i) = diag(ShockCov(:,:,i));
   MeasureVar(:,i) = diag(MeasureCov);
   ParameterVar(:,i) = diag(ParameterCov(:,:,i));
end;
%
% compute the total
%
TotalVar = StateVar+ShockVar+MeasureVar+ParameterVar;
for i=1:h;
   for j=1:n;
      if TotalVar(j,i)<=1.0e-010;
         TotalVar(j,i) = 1;
         StateVar(j,i) = 0;
         ShockVar(j,i) = 0;
         MeasureVar(j,i) = 0;
         ParameterVar(j,i) = 0;
      end;
   end;
end;
%
% and now the shares
%
StateVar = 100*(StateVar ./ TotalVar);
ShockVar = 100*(ShockVar ./ TotalVar);
MeasureVar = 100*(MeasureVar ./ TotalVar);
ParameterVar = 100*(ParameterVar ./ TotalVar);

%
% end of SummarizeDSGEPredictions.m
%
