function GewekeStat = SeparatedPartialMeansTest(PostSample,p,PlotType,CurrINI)
% SeparatedPartialMeansTest: Computes the separated partial means test suggested by Geweke.
%
% USAGE:
%
%       GewekeStat = SeparatedPartialMeansTest(PostSample,p,PlotType,CurrINI)
%
% REQUIRED INPUT:  PostSample (matrix) of dimension NumIter x NumParam, where NumIter is the
%                             number of draws and NumParam the number of parameters.
%
%                  p (integer) with the number of separated partial means. The ratio
%                             Np = NumIter/2p must also be an integer.
%
%                  PlotType (string vector) that takes on the values "log posterior",
%                             "original parameters", and "transformed parameters".
%
%                  CurrINI (structure) with initialization information.
%
% REQUIRED OUTPUT: GewekeStat (matrix) of dimension NumParam x 2 where the first column
%                             holds the values of the NumParam different separated partial
%                             means tests, while the second column holds the p-values from
%                             the chi-square distribution with p-1 degrees of freedom.
%
% REFERENCE:  Geweke, J. (2005), Contemporary Bayesian Econometrics and Statistics,
%                             John Wiley: Hoboken, NJ.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: April 19, 2007.
%                        This version: January 22, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 20-04-2007: Fixed a bug where the non-zero off-diagonal (j,j+1) and (j+1,j) elements of v_p where set to tau_p(j,1)
%               instead of tau_p(j+1,1).
%
% * 13-11-2007: Updated the documentation.
%
% * 10-01-2008: Changed the last drawnow call to pause(0.02).
%
% * 23-05-2008: Updated the documentation.
%
% * 21-07-2008: Added check for the existence of relevant chi-square distribution functions.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
GewekeStat = [];
[NumIter NumParam] = size(PostSample);
Np = NumIter/(2*p);
if IsInteger(Np)==0;
   return;
end;
GewekeStat = zeros(NumParam,2);
%
% set up a wait dialog
%
txt = ['Please wait while YADA computes Geweke''s separated partial means tests for the ' lower(PlotType) ' using ' num2str(p,'%0.0f') ' partial means and ' num2str(NumIter,'%0.0f') ' posterior draws. Estimation started at: ' StringTrim(datestr(now,14)) '.'];
WaitHandle = WaitDLG(txt,'information',['Separated Partial Means Test - ' PlotType],500,200,CurrINI,1);
WaitControls = get(WaitHandle,'UserData');
drawnow;
%
StopEstimation = 0;
for i=1:NumParam;
   if strcmp(get(WaitControls.cancel,'UserData'),'cancel')==1;
      StopEstimation = 1;
      break;
   end;
   %
   % create the partial means for j=1,...,p
   %
   s_p = zeros(p,1);
   tau_p = zeros(p,1);
   for j=1:p;
      %
      % compute the separated means
      %
      s_p(j,1) = mean(PostSample(1+(Np*((2*j)-1)):Np+(Np*((2*j)-1)),i));
      drawnow;
      if strcmp(get(WaitControls.cancel,'UserData'),'cancel')==1;
         StopEstimation = 1;
         break;
      end;
      %
      % compute the Newey-West autocorrelation consistent squared numerical standard errors.
      %
      tau_p(j,1) = NeweyWestCovMat(PostSample(1+(Np*((2*j)-1)):Np+(Np*((2*j)-1)),i),floor(Np^(1/2.01)));
      drawnow;
      if strcmp(get(WaitControls.cancel,'UserData'),'cancel')==1;
         StopEstimation = 1;
         break;
      end;
   end;
   if StopEstimation==1;
      break;
   end;
   %
   % compute the separated partial means
   %
   h_p = diff(s_p);
   %
   % compute the covariance matrix
   %
   v_p = zeros(p-1,p-1);
   for j=1:p-1;
      v_p(j,j) = tau_p(j,1)+tau_p(j+1,1);
      if j<p-1;
         v_p(j,j+1) = tau_p(j+1,1);
         v_p(j+1,j) = tau_p(j+1,1);
      end;
   end;
   %
   % compute the test-statistic
   %
   GewekeStat(i,1) = h_p'*inv(v_p)*h_p;
   drawnow;
end;
%
% shut down the wait dialog
%
set(WaitControls.text,'UserData','done');
delete(WaitHandle);
drawnow;
pause(0.02);
%
if StopEstimation==1;
   GewekeStat = [];
   return;
end;
%
% compute the p-values
%
if exist('YADAchi2cdf')==2;
   GewekeStat(:,2) = ones(NumParam,1)-YADAchi2cdf(GewekeStat(:,1),(p-1)*ones(NumParam,1));
else;
   if exist('chi2cdf')==2;
      GewekeStat(:,2) = ones(NumParam,1)-chi2cdf(GewekeStat(:,1),(p-1)*ones(NumParam,1));
   else;
      GewekeStat(:,2) = ones(NumParam,1)-YADACheapchi2cdf(GewekeStat(:,1),(p-1)*ones(NumParam,1));
   end;
end;

%
% end of SeparatedPartialMeansTest.m
%
