function [MeanPrediction,Quantiles] = PredictionPathDistribution(PredPaths,QuantileVector)
% PredictionPathDistribution: Computes the mean value of prediction paths as well as quantiles.
%
% USAGE:
%
%       [MeanPrediction,Quantiles] = PredictionPathDistribution(PredPaths,QuantilesVector)
%
% REQUIRED INPUT:  PredPaths (matrix) of dimension nxhxNumPaths, where n is the number of predicted variables,
%                            h the prediction horizon, and NumPaths the number of predicted paths.
%
%                  QuantileVector (vector) of dimension 1xNumQuants with percent values for the quantiles,
%                            e.g. [5 10 90 95].
%
% REQUIRED OUTPUT: MeanPrediction (matrix) of dimension nxh with the mean predictions.
%
%                  Quantiles (structure) of dimension NumQuants where each has a field called "Quantile" which
%                            is an nxh matrix with the quantiles of the prediction paths. The second field
%                            is "percent" that gives the percentile value.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: February 19, 2007.
%                        This version: January 22, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 21-02-2007: The field "percent" is now an integer instead of a string.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% Initialize
%
[n,h,NumPaths] = size(PredPaths);
NumQuants = length(QuantileVector);
%
% compute the mean prediction
%
MeanPrediction = zeros(n,h);
for i=1:NumPaths;
   MeanPrediction = MeanPrediction+((1/NumPaths)*PredPaths(:,:,i));
end;
%
% Initialize quantiles output
%
for i=1:NumQuants;
   Quantiles(i).Quantile = zeros(n,h);
   Quantiles(i).percent = QuantileVector(i);
end;
%
% compute all the quantile values
%
for j=1:h;
   for i=1:n;
      TempVar = zeros(NumPaths,1);
      for p=1:NumPaths;
         TempVar(p,1) = PredPaths(i,j,p);
      end;
      TempVar = sort(TempVar);
      for k=1:NumQuants;
         if QuantileVector(k)<50;
            QuantVal = ceil((QuantileVector(k)/100)*NumPaths);
         else;
            QuantVal = floor((QuantileVector(k)/100)*NumPaths);
         end;
         Quantiles(k).Quantile(i,j) = TempVar(QuantVal,1);
      end;
   end;
end;

%
% end of PredictionPathDistribution.m
%