function ParData = ParameterSummary(PostSample)
% ParameterSummary: Computes summary statistics about the posterior sample.
%
% USAGE:
%
%       ParData = ParameterSummary(PostSample)
%
% REQUIRED INPUT:  PostSample (matrix) with dimension NumIterxNumParam.
%
% REQUIRED OUTPUT: ParData (structure), with fields "SampleMean" containing the sample
%                          mean of the different parameters, "SampleStd" with the sample
%                          standard deviations, and "SampleMedian" with the sample median
%                          parameter values. The field "SampleStdErr" contains numerical
%                          standard errors of the SampleMean estimate. The field "SampleStdAuto"
%                          contains the autocorrelation consistent standard deviations. The
%                           "Quantiles" field contains quantile values of the parameters.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: December 8, 2006.
%                        This version: January 22, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-12-2006: Added the "Quantiles" field.
%
% * 22-12-2006: The drawnow command after the call to the "NeweyWestCovMat" function is now executed
%               for each parameter rather than after the full loop. A drawnow command was also added
%               to the "Quantiles" loop.
%
% * 23-02-2007: All calculations now loop over the number of parameters. This demands less memory.
%
% * 12-04-2007: Changed the BarN value to floor(NumIter^(1/2.01)) when calculation Newey-West standard errors.
%
% * 19-04-2007: Added the field SampleStdAuto with autocorrelation corrected standard deviations.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

ParData = [];
[NumIter,NumParam] = size(PostSample);
ParData.SampleMean = zeros(1,NumParam);
ParData.SampleStd = zeros(1,NumParam);
ParData.SampleMedian = zeros(1,NumParam);
for i=1:NumParam;
   ParData.SampleMean(1,i) = mean(PostSample(:,i));
   if NumIter>=40000;
      drawnow;
   end;
end;
for i=1:NumParam;
   ParData.SampleStd(1,i) = std(PostSample(:,i));
   if NumIter>=40000;
      drawnow;
   end;
end;
ParData.SampleStdErr = zeros(1,NumParam);
ParData.SampleStdAuto = zeros(1,NumParam);
for i=1:NumParam;
   ParData.SampleStdErr(i) = sqrt(NeweyWestCovMat(PostSample(:,i),floor(NumIter^(1/2.01))));
   ParData.SampleStdAuto(i) = sqrt(NumIter*ParData.SampleStdErr(i)^2);
   if NumIter>=40000;
      drawnow;
   end;
end;
for i=1:NumParam;
   ParData.SampleMedian(1,i) = median(PostSample(:,i));
   if NumIter>=40000;
      drawnow;
   end;
end;
ParData.Quantiles = zeros(NumParam,9);
for i=1:NumParam;
   ParVec = sort(PostSample(:,i));
%   q_025 = ceil(0.025*NumIter);
%   q_05 = ceil(0.05*NumIter);
%   q_10 = ceil(0.1*NumIter);
%   q_20 = ceil(0.2*NumIter);
%   q_50 = ceil(0.5*NumIter);
%   q_80 = floor(0.8*NumIter);
%   q_90 = floor(0.9*NumIter);
%   q_95 = floor(0.95*NumIter);
%   q_975 = floor(0.975*NumIter);
   ParData.Quantiles(i,1) = ParVec(ceil(0.025*NumIter));
   ParData.Quantiles(i,2) = ParVec(ceil(0.05*NumIter));
   ParData.Quantiles(i,3) = ParVec(ceil(0.1*NumIter));
   ParData.Quantiles(i,4) = ParVec(ceil(0.2*NumIter));
   ParData.Quantiles(i,5) = ParVec(ceil(0.5*NumIter));
   ParData.Quantiles(i,6) = ParVec(floor(0.8*NumIter));
   ParData.Quantiles(i,7) = ParVec(floor(0.9*NumIter));
   ParData.Quantiles(i,8) = ParVec(floor(0.95*NumIter));
   ParData.Quantiles(i,9) = ParVec(floor(0.975*NumIter));
   if NumIter>=40000;
      drawnow;
   end;
end;

%
% end of ParameterSummary.m
%
