function StdErr = NeweyWestCovMat(PostSample,BarN)
% NeweyWestCovMat: Computes Newey-West standard errors.
%
% USAGE:
%
%       StdErr = NeweyWestCovMat(PostSample,BarN)
%
% REQUIRED INPUT:  PostSample (matrix) of dimension NumDraws x NumVars.
%
% OPTIONAL INPUT:  BarN (integer), with the number of autocorrelations to take into account. If BarN is not
%                       specified or is empty, the assigned value is NumDraws^(1/2.01) unless NumDraws^(1/2.01)<100
%                       and NumDraws>200. In that case BarN=100.
%
% REQUIRED OUTPUT: StdErr (matrix) of dimension NumVars with the covariance matrix for computing the
%                       Newey-West standard errors as the square root of its diagonal.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: September 1, 2006.
%                        This version: January 22, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 30-11-2006: Changed the output from the standard errors to the covariance matrix.
%
% * 08-12-2006: Fixed a bug such that CovMati is taken into account both at leads and at lags.
%
% * 02-02-2007: Added a check for Inf entries.
%
% * 12-04-2007: Changed the default "BarN" value to N^(1/2.01). This means that lim(N->infinity) BarN = +infinity
%               while lim(N->infinity) BarN^2/2 = 0. See, e.g., Theorem 4.7.3 in Geweke (2005).
%
% * 13-11-2007: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% check for Inf entries in PostSample
%
if max(max(isinf(PostSample)))==1;
   StdErr = [];
   return;
end;
%
[N,m] = size(PostSample);
if (nargin==1)|(isempty(BarN)==1)|(floor(BarN)<1);
   BarN = floor(N^(1/2.01));
   if (BarN<100)&(N>200);
      BarN = 100;
   end;
else;
   %
   % make sure it's an integer
   %
   BarN = round(BarN);
end;
%
% First compute the mean of PostSample
%
mu = sum(PostSample)/N;
%
% mean adjust PostSample
%
PostSample = PostSample-(ones(N,1)*mu);
CovMat = (PostSample'*PostSample/N);
%
CovMatSum = zeros(size(CovMat));
for i=1:BarN;
   CovMati = (PostSample(1+i:N,:)'*PostSample(1:N-i,:))/N;
   CovMatSum = CovMatSum+(((BarN+1-i)/(BarN+1))*CovMati);
end;
CovMat = (CovMat+CovMatSum+CovMatSum')/N;
%
% Test if CovMat is well behaved
%
TestValue = sum(log(eig(CovMat)));
if (isinf(TestValue)==0)&(isnan(TestValue)==0)&(isreal(TestValue)==1);
   StdErr = CovMat;
else;
   StdErr = NaN*ones(m,m);
end;

%
% end of NeweyWestCovMat.m
%
