function [JointProb,ShockProb] = ImplausibilityIndexFcn(ShockMatrix)
% ImplausibilityIndexFcn: Computes the joint and individual probabilities for the
%                         implausibility index.
%
% USAGE:
%
%       [JointProb,ShockProb] = ImplausibilityIndexFcn(ShockMatrix)
%
% REQUIRED INPUT:  ShockMatrix (matrix) of dimension nxT of shocks that should be iid N(0,I)
%                              if they are drawn from the model based distribution of the shocks.
%
% REQUIRED OUTPUT: JointProb (vector) of dimension T with the joint probability that the shocks
%                              are iid normal using the chi-squared form.
%
%                  ShockProb (matrix) of dimension nxT that the individuals shocks are iid normal
%                              using the chi-squared form.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: August 27, 2008.
%                        This version: January 22, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 28-08-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
[n,T] = size(ShockMatrix);
JointProb = zeros(1,T);
ShockProb = zeros(n,T);
%
JointValue = zeros(1,T);
ShockValues = zeros(n,T);
for t=1:T;
   if t>1;
      JointValue(1,t) = JointValue(1,t-1)+(ShockMatrix(:,t)'*ShockMatrix(:,t));
   else;
      JointValue(1,1) = ShockMatrix(:,1)'*ShockMatrix(:,t);
   end;
   for j=1:n;
      if t>1;
         ShockValues(j,t) = ShockValues(j,t-1)+(ShockMatrix(j,t)^2);
      else;
         ShockValues(j,1) = (ShockMatrix(j,1)^2);
      end;
   end;
end;
if exist('YADAchi2cdf')==2;
   JointProb = YADAchi2cdf(JointValue,(n:n:n*T));
   for j=1:n;
      ShockProb(j,:) = YADAchi2cdf(ShockValues(j,:),(1:T));
   end;
else;
   if exist('chi2cdf')==2;
      JointProb = chi2cdf(JointValue,(n:n:n*T));
      for j=1:n;
         ShockProb(j,:) = chi2cdf(ShockValues(j,:),(1:T));
      end;
   else;
      JointProb = YADACheapchi2cdf(JointValue,(n:n:n*T));
      for j=1:n;
         ShockProb(j,:) = YADACheapchi2cdf(ShockValues(j,:),(1:T));
      end;
   end;
end;

%
% end of ImplausibilityIndexFcn.m
%
