function [status,EigenValues] = DSGEtoVARModel(H,R,F,B0)
% DSGEtoVARModel: Checks if the state-space representation of the DSGE model can be rewritten
%                as an infinite order VAR model.
%
% USAGE:
%
%       [status,EigenValues] = DSGEtoVARModel(H,R,F,B0)
%
% REQUIRED INPUT:  H (rxn) matrix with cofficients on the state variables in the measurement equation.
%
%                  R (nxn) matrix with the covariances of the measurement errors.
%
%                  F (rxr) transition matrix for the state variables.
%
%                  B0 (rxq) matrix with coefficients on the economic shocks in the state equation.
%
% REQUIRED OUTPUT: status (integer) that takes on the value 1 if the DSGE model can be rewritten as
%                         and infinite order VAR; 0 if the invertibility condition of Fernandez-Villaverde,
%                         Rubio-Ramirez, Sargent and Watson (AER, 2007) is not satisfied; and -1 if the
%                         number of economic shocks and unique measurement errors is greater than the
%                         number of observed variables.
%
%                  EigenValues (vector) with the modulus of the eigenvalues from the "poor man's
%                         invertibility condition".
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: October 11, 2007.
%                        This version: January 22, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 15-10-2007: Changed the function name from "DSGEToVARModel" to "DSGEtoVARModel".
%
% * 30-10-2007: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

status = 0;
EigenValues = [];
%
% determine certain dimensional parameters
%
[r,n] = size(H);
%
% first we decompose R into a nxm matrix Phi such that R = Phi*Phi'
%
if rank(R)>0;
   [Rm,Lm] = SchurDecomposition(R);
   Phi = Rm*sqrt(Lm);
   m = size(Lm,1);
   ZeroMatrix = zeros(n,m);
else;
   Phi = [];
   ZeroMatrix = [];
   m = 0;
end;
%
% create the B and D matrices
%
[B0,KeepShocks] = RemoveRedundantColumns(B0);
q = size(B0,2);
if n<q+m;
   %
   % too many economic shocks and measurement errors in relation to the number of observed variables
   %
   status = -1;
   return;
end;
%
% check again the number of economic shocks and measurement errors in relation to the number of
% observed variables
%
D = [H'*B0 Phi];
if rank(D)<m+q;
   %
   % too many economic shocks and measurement errors in relation to the number of observed variables
   %
   status = -1;
   return;
end;
%
% we can check the invertibility condition
%
B = [B0 ZeroMatrix];
EigenValues = abs(eig(F-(B*inv(D'*D)*D'*H'*F)));
if max(EigenValues)<1-(1.0e-10);
   status = 1;
end;

%
% end of DSGEtoVARModel.m
%
