function [MeanPrediction,Quantiles,KernelX,KernelY] = DSGEPredictionDistribution(n,OutputDirectory,CondStr,ShockMethod,NameOfModel,MaxPH,NumDraws,NumPredPaths,AnnualizeData,TransData,CurrChain,QuantileVector,VariableNames,CurrINI)
% DSGEPredictionDistribution: Computes the mean value of conditional and unconditional prediction paths
%                             as well as quantiles by loading data from disk.
%
% USAGE:
%
%       [MeanPrediction,Quantiles,KernelX,KernelY] = DSGEPredictionDistribution(n,OutputDirectory,CondStr,ShockMethod,...
%                                                         NameOfModel,MaxPH,NumDraws,NumPredPaths,AnnualizeData,TransData,...
%                                                         CurrChain,QuantileVector,VariableNames,CurrINI)
%
% REQUIRED INPUT:  n (integer) with the number of observed variables.
%
%                  OutputDirectory (string vector) with the full path to the base output directory.
%
%                  CondStr (string vector) that is either "Cond" or "".
%
%                  ShockMethod (string vector) that takes on the values "Direct-", "WaggonerZha-" or "".
%
%                  NameOfModel (string vector) with the name of the model.
%
%                  MaxPH (integer) with the length of the prediction horizon.
%
%                  NumDraws (integer) with the total number of parameter draws.
%
%                  NumPredPaths (integer) with tht total number of prediction paths per parameter draw.
%
%                  AnnualizeData (boolean) that take the value 1 if annualized results should be checked and 0 otherwise.
%
%                  TransData (boolean) that is 1 if the prediction paths should be transformed and 0 otherwise.
%
%                  CurrChain (integer) with the number of the currently used Markov chain.
%
%                  QuantileVector (vector) of dimension 1xNumQuants with percent values for the quantiles,
%                            e.g. [5 10 90 95].
%
%                  VariableNames (string matrix) with the names of the observed variables.
%
%                  CurrINI (structure) with initialization information.
%
% REQUIRED OUTPUT: MeanPrediction (matrix) of dimension nxh with the mean predictions.
%
%                  Quantiles (structure) of dimension NumQuants where each has a field called "Quantile" which
%                            is an nxh matrix with the quantiles of the prediction paths. The second field
%                            is "percent" that gives the percentile value.
%
%                  KernelX (3D matrix) of dimension n x 2^8 x MaxPH with the x-axis values.
%
%                  KernelY (3D matrix) of dimension n x 2^8 x MaxPH with the y-axis values.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: February 23, 2007.
%                        This version: March 3, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 12-03-2007: Added the input "CondStr" to handle conditional predictive distributions.
%
% * 13-11-2007: Updated the documentation.
%
% * 19-12-2007: Added the input "ShockMethod".
%
% * 20-12-2007: Hunting an annualization bug.
%
% * 10-01-2008: Changed the last drawnow call to pause(0.02).
%
% * 05-05-2008: Added the input variable "TransData".
%
% * 23-05-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
% * 03-03-2009: Added the output variables "KernelX" and "KernelY".
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
MeanPrediction = zeros(n,MaxPH);
NumQuants = length(QuantileVector);
KernelX = zeros(n,2^8,MaxPH);
KernelY = KernelX;
%
% Initialize quantiles output
%
for i=1:NumQuants;
   Quantiles(i).Quantile = zeros(n,MaxPH);
   Quantiles(i).percent = QuantileVector(i);
end;
%
if isempty(CondStr)==1;
   HeadStr = 'Predictive Distributions';
   txtStr = '';
else;
   HeadStr = 'Conditional Predictive Distributions';
   txtStr = 'conditional ';
end;
%
% set up a wait dialog
%
txt = ['Please wait while YADA computes the mean ' txtStr 'predictions for the observed variables. The data are loaded sequentially from disk and it may therefore take some time.'];
WaitHandle = WaitDLG(txt,'information',HeadStr,500,150,CurrINI,0);
WaitControls = get(WaitHandle,'UserData');
%
% compute the mean prediction
%
TotalPaths = NumPredPaths*NumDraws;
for ParamIndex=1:NumDraws;
   if AnnualizeData==0;
      if TransData==0;
         file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Original-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];
      else;
         file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Transformed-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];
      end;
   else;
      file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Annual-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];
   end;
   %
   % load the data
   %
   PredData = load(file);
   PredPaths = PredData.PredPaths;
   for i=1:NumPredPaths;
      MeanPrediction = MeanPrediction+((1/TotalPaths)*PredPaths(:,:,i));
   end;
   clear('PredData');
   drawnow;
end;
%
% compute all the quantile values
%
for j=1:MaxPH;
   for i=1:n;
      txt = ['Please wait while YADA computes quantile values and kernel density estimates for the ' txtStr 'predictions of the observed variables. Current variable: ' StringTrim(VariableNames(i,:)) ' and horizon ' num2str(j,'%0.0f') '.'];
      set(WaitControls.text,'String',txt);
      drawnow;
      %
      % initialzie a temporary variable
      %
      TempVar = zeros(TotalPaths,1);
      for ParamIndex=1:NumDraws;
         if AnnualizeData==0;
            if TransData==0;
               file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Original-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];
            else;
               file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Transformed-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];
            end;
         else;
            file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Annual-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];
         end;
         %
         % load the data
         %
         PredData = load(file);
         PredPaths = PredData.PredPaths;
         for p=1:NumPredPaths;
            CurrPath = p+(NumPredPaths*(ParamIndex-1));
            TempVar(CurrPath,1) = PredPaths(i,j,p);
         end;
         clear('PredData');
         drawnow;
      end;
      %
      % TempVar is now fully loaded and may
      %
      TempVar = sort(TempVar);
      for k=1:NumQuants;
         if QuantileVector(k)<50;
            QuantVal = ceil((QuantileVector(k)/100)*TotalPaths);
         else;
            QuantVal = floor((QuantileVector(k)/100)*TotalPaths);
         end;
         Quantiles(k).Quantile(i,j) = TempVar(QuantVal,1);
      end;
      %
      if TempVar(1)+1.0e-8<TempVar(TotalPaths);
         [x,f] = YADAMHDensEst(TempVar,2^8,0,'gaussian',[]);
         KernelX(i,:,j) = x';
         KernelY(i,:,j) = f';
      end;
   end;
end;
%
% close the wait dialog
%
set(WaitControls.text,'UserData','done');
delete(WaitHandle);
drawnow;
pause(0.02);

%
% end of DSGEPredictionDistribution.m
%
