function [MultiPValue,MeanMultiModesty,MeanUniModesty,MeanUniModestyLZ] = DSGEModestyStatistics(OutputDirectory,ShockMethod,NameOfModel,MaxPH,NumDraws,NumPredPaths,CurrChain,n)
% DSGEModestyStatistics: Computes the modesty statistics.
%
% USAGE:
%
%       [MultiPValue,MeanMultiModesty,MeanUniModesty,MeanUniModestyLZ] = DSGEModestyStatistics(OutputDirectory,ShockMethod,NameOfModel,MaxPH,NumDraws,NumPaths,CurrChain,n)
%
% REQUIRED INPUT:  OutputDirectory (string vector) with the full path to the base output directory.
%
%                  ShockMethod (string vector) that takes on the value "Direct-" or "WaggonerZha-".
%
%                  NameOfModel (string vector) with the name of the model.
%
%                  MaxPH (integer) with the length of the prediction horizon.
%
%                  NumDraws (integer) with the total number of parameter draws.
%
%                  NumPredPaths (integer) with tht total number of prediction paths per parameter draw.
%
%                  CurrChain (integer) with the number of the currently used Markov chain.
%
%                  n (integer) with the number of observed variables.
%
% REQUIRED OUTPUT: MultiPValue (scalar) with the probability that the multivariate modesty statistic
%                              is smaller than one based on unconditional shocks.
%
%                  MeanMultiModesty (vector) with 2 elements. The first is the mean of the multivariate
%                              modesty statistic based on the conditioning consistent shocks and the second
%                              is the mean of this statistic for randomly drawn shocks.
%
%                  MeanUniModesty (vector) with means of the univariate modesty statistics for all observed
%                              variables.
%
%                  MeanUniModestyLZ (vector) with the means of the univariate modesty statistics suggested
%                              by Leeper-Zha (other shocks are zero over examined prediction horizon) for
%                              all observed variables.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: March 12, 2007.
%                        This version: January 22, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-03-2007: Fixed a silly bug when computing the mean of "UniModestyStatLZ".
%
% * 13-11-2007: Updated the documentation.
%
% * 19-12-2007: Added the input "ShockMethod".
%
% * 23-05-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

TotalPaths = NumPredPaths*NumDraws;
MultiModestyStat = zeros(TotalPaths,2);
UniModestyStat = zeros(TotalPaths,n);
UniModestyStatLZ = zeros(NumDraws,n);
LoadIndex = (1:NumPredPaths);
for ParamIndex=1:NumDraws;
   file = [OutputDirectory '\predictions\DSGE-CondPrediction-Original-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];;
   %
   % load the data
   %
   PredData = load(file);
   MultiModestyStat(LoadIndex,:) = PredData.MultiModestyStat;
   UniModestyStat(LoadIndex,:) = PredData.UniModestyStat;
   UniModestyStatLZ(ParamIndex,:) = PredData.UniModestyStatLZ;
   clear('PredData');
   drawnow;
   LoadIndex = LoadIndex+NumPredPaths;
end;
%
% check how often we have a greater value of column 1 than column 2 in MultiModestyStat
%
MultiPValue = sum(MultiModestyStat(:,1)<=MultiModestyStat(:,2))/TotalPaths;
MeanMultiModesty = (1/TotalPaths)*[sum(MultiModestyStat(:,1)) sum(MultiModestyStat(:,2))];
MeanUniModesty = zeros(1,n);
MeanUniModestyLZ = zeros(1,n);
for i=1:n;
   MeanUniModesty(1,i) = (1/TotalPaths)*sum(UniModestyStat(:,i));
   MeanUniModestyLZ(1,i) = (1/NumDraws)*sum(UniModestyStatLZ(:,i));
end;

%
% end of DSGEModestyStatistics.m
%
