function DoneCalc = DSGEForecastErrorDecomposition(theta,thetaPostSample,thetaPositions,ModelParameters,DSGEModel,CurrINI,h,TotalDraws,CurrChain)
% DSGEForecastErrorDecomposition: Calculates historical decompositions of the h-steps ahead forecast errors using a subset of the
%                                 posterior draws.
%
% USAGE:
%
%       DoneCalc = DSGEForecastErrorDecomposition(theta,thetaPostSample,thetaPositions,ModelParameters,DSGEModel,CurrINI,h,TotalDraws,CurrChain)
%
% REQUIRED INPUT:  theta (vector) with the posterior mode estimates of the parameters.
%
%                  thetaPostSample (matrix) of dimension NumDrawsxNumParam with draws from the posterior distribution
%                                  of the original DSGE model parameters.
%
%                  thetaPositions (structure) with fields parameter and distribution. This function
%                                 needs the former field to know which parameter name corresponds to
%                                 position i in theta.
%
%                  ModelParameters (structure) whose fields have names given by the parameter names in the
%                                 DSGE model.
%
%                  DSGEModel (structure) with fields containing all the information about a DSGE Model for YADA.
%
%                  CurrINI (structure), with initialization information.
%
%                  h (positive integer), with the forecast horizon.
%
%                  TotalDraws (integer) with the total number of parameter draws that could have been used.
%
%                  CurrChain (integer) with the number for the current Markov chain.
%
% REQUIRED OUTPUT: DoneCalc (boolean) that is 1 if the calculations were completed and 0 otherwise.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: May 7, 2008.
%                        This version: March 9, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 23-05-2008: Updated the documentation.
%
% * 29-05-2008: Added a test to check if the model can be solved at the mode.
%
% * 02-07-2008: Added the variable "DSGEModel.AllowUnitRoot" for the Kalman filters.
%
% * 07-07-2008: Added code to allow for the case when the measurement matrix H is time-varying.
%
% * 22-01-2009: Updated the documentation.
%
% * 13-02-2009: The mcode values 7 and 8 are now properly mentioned when encountered.
%
% * 09-03-2009: Added quotation marks around paths for the dos copy commands.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% intialize the output
%
DoneCalc = 0;
NumDraws = size(thetaPostSample,1);
thetaMode = theta;
%
% copy files to the tmp folder
%
[stat,msg] = dos(['copy /Y "' GetPath(DSGEModel.AIMDataFile) 'compute_aim_matrices.m" "' pwd '\tmp"']);
[stat,msg] = dos(['copy /Y "' DSGEModel.MeasurementEquationFile '" "' pwd '\tmp"']);
if FileExist(DSGEModel.UpdateParameterFile)==1;
   [stat,msg] = dos(['copy /Y "' DSGEModel.UpdateParameterFile '" "' pwd '\tmp"']);
end;
%
% load the AIMData file
%
ErrorStr = '';
try;
   AIMData = load(DSGEModel.AIMDataFile);
catch;
   ErrorStr = ['Unable to load the file "' DSGEModel.AIMDataFile '". Message caught is: ' lasterr];
end;
if isempty(ErrorStr)==0;
   About(ErrorStr,'information','YADA - Bad MAT File',120,500,CurrINI);
   return;
end;
%
% create the output directory
%
stat = MakeDir(DSGEModel.OutputDirectory,'hfed');
if stat~=1;
   txt = ['YADA was for some reason unable to create the directory "' DSGEModel.OutputDirectory '\hfed". The computation distributions for the historical forecast error decompositions has therefore been aborted.'];
   About(txt,'information','YADA - Directory Creation Problem',180,500,CurrINI);
   return;
end;
%
% determine the sample of data
%
[FirstPeriod,LastPeriod] = CreateSubSample(DSGEModel);
Y = DSGEModel.Y(:,FirstPeriod+DSGEModel.KalmanFirstObservation-1:LastPeriod);
if isempty(DSGEModel.X)==0;
   X = DSGEModel.X(:,FirstPeriod+DSGEModel.KalmanFirstObservation-1:LastPeriod);
else;
   X = [];
end;
T = LastPeriod-FirstPeriod+1;
Ttot = size(Y,2);
%
% set the InitialStateVector value
%
if DSGEModel.UseOwnInitialState==1;
   if length(DSGEModel.InitialStateValues)==size(AIMData.endog,1);
      DSGEModel.InitialStateVector = DSGEModel.InitialStateValues(DSGEModel.StateVariablePositions);
   else;
      DSGEModel.InitialStateVector = zeros(length(DSGEModel.StateVariablePositions),1);
   end;
else;
   DSGEModel.InitialStateVector = zeros(length(DSGEModel.StateVariablePositions),1);
end;
%
% solve the model for the posterior mode estimate
%
ErrorStr = '';
try;
   [A,H,R,F,B0,mcode] = SolveDSGEModel(DSGEModel,theta,thetaPositions,ModelParameters,AIMData);
catch;
   ErrorStr = ['YADA caught an error when trying to solve the DSGE model. Message caught is:' lasterr];
end;
if isempty(ErrorStr)==0;
   About(ErrorStr,'information','YADA - Error Message',200,500,CurrINI);
   return;
end;
if mcode~=1;
   if mcode==2;
      mcodeStr = 'Roots are not correctly computed by real_schur.'
   elseif mcode==3;
      mcodeStr = 'Too many big roots.';
   elseif mcode==35;
      mcodeStr = 'Too many big roots, and q(:,right) is singular.';
   elseif mcode==4;
      mcodeStr = 'Too few big roots.';
   elseif mcode==45;
      mcodeStr = 'Too few big roots, and q(:,right) is singular.';
   elseif mcode==5;
      mcodeStr = 'q(:,right) is singular.';
   elseif mcode==61;
      mcodeStr = 'Too many exact shiftrights.';
   elseif mcode==62;
      mcodeStr = 'Too many numeric shiftrights.';
   elseif mcode==7;
      mcodeStr = 'Infinite or NaN values detected.';
   elseif mcode==8;
      mcodeStr = 'The function "compute_aim_matrices" returns complex numbers.';
   else;
      mcodeStr = 'Return code not properly specified.';
   end;
   txt = ['The AiM solver provided the return code: ' num2str(mcode) ', i.e., "' mcodeStr '"'];
   About(txt,'information','YADA - AiM Solver Error',200,500,CurrINI);
   return;
end;
if length(size(H))==3;
   if size(H,3)<LastPeriod;
      %
      % too few data points in the 3D H matrix.
      %
      ErrorStr = ['The number of time periods for the time-varying measurement matrix H (' int2str(size(H,3)) ') is lower than the number of observations (T = ' int2str(LastPeriod) '). YADA has therefore aborted from the forecast error decompositions.'];
      About(ErrorStr,'error','Error - Measurement Equation',200,500,CurrINI);
      return;
   end;
end;
%
[B0,KeepShocks] = RemoveRedundantColumns(B0);
Rmode = R;
%
% set up the date vector
%
DateVector = MakeDateVector(DSGEModel.BeginYear,DSGEModel.BeginPeriod,DSGEModel.EndYear,DSGEModel.EndPeriod,DSGEModel.DataFrequency);
DateVector = DateVector(FirstPeriod+DSGEModel.KalmanFirstObservation-1+h:LastPeriod);
%
% initialize index for saving
%
SaveAfterDraws = min(NumDraws,100);
NumFiles = ceil(NumDraws/SaveAfterDraws);
q = length(DSGEModel.StateShockPositions);
n = DSGEModel.n;
r = length(DSGEModel.StateVariablePositions);
%
% begin the loop
%
abort = '';
MeanEstimationTime = 0;
LastDraw = 0;
ThisSave = 0;
HFED = [];
%
% check if we should setup a wait handle
%
if DSGEModel.ShowProgress==1;
   ProgressStructure.title = ['Progress for ' num2str(NumDraws,'%0.0f') ' parameter draws'];
   ProgressStructure.facecolor = CurrINI.progress_facecolor;
   ProgressStructure.startfacecolor = CurrINI.progress_startfacecolor;
   ProgressStructure.edgecolor = CurrINI.progress_edgecolor;
   ProgressStructure.bgcolor = CurrINI.progress_bgcolor;
   ProgressStructure.stop = 0;
   ProgressStructure.clock = DSGEModel.ShowProgressClock;
   ProgressStructure.label = 'Mean estimation time:';
   %
   ProgressStructure.name = 'Forecast Error Decomposition';
   ProgressStructure.CurrINI = CurrINI;
   WaitHandle = ProgressDLG(0,ProgressStructure);
   set(WaitHandle,'Color',get(0,'defaultuicontrolbackgroundcolor'));
   drawnow;
else;
   txt = ['Please wait while YADA computes the distribution of the historical forecast error decompsotions. Computations started at: ' StringTrim(datestr(now,14))];
   WaitHandle = WaitDLG(txt,'information','Forecast Error Decomposition',500,150,CurrINI,0);
   WaitControls = get(WaitHandle,'UserData');
end;
%
% start the iterations
%
if length(size(H))==2;
   for it=1:NumDraws;
      LastDraw = LastDraw+1;
      if DSGEModel.ShowProgress==1;
         abort = get(WaitHandle,'UserData');
         if strcmp(abort,'cancel')==1;
            break;
         else;
            ProgressDLG([it/NumDraws MeanEstimationTime]);
         end;
      end;
      %
      % Solve the model for the current parameter vector
      %
      theta = thetaPostSample(it,:)';
      tic;
      [A,H,R,F,B0,status] = SolveDSGEModel(DSGEModel,theta,thetaPositions,ModelParameters,AIMData);
      if status~=1;
         %
         % use posterior mode estimate as a backup
         %
         [A,H,R,F,B0,status] = SolveDSGEModel(DSGEModel,thetaMode,thetaPositions,ModelParameters,AIMData);
      end;
      B0 = B0(:,KeepShocks);
      %
      % run the Kalman filter to get the estimate of the state forecasts
      %
      if isempty(DSGEModel.UnitRootStates)==1;
         if isempty(DSGEModel.X)==0;
            [logLikeValue,stat,lnLt,Yhat,MSEY,Ksitt1,Ptt1] = KalmanFilter(DSGEModel.Y(:,FirstPeriod:LastPeriod),DSGEModel.X(:,FirstPeriod:LastPeriod),A,H,F,B0*B0',R,DSGEModel.InitialStateVector,DSGEModel.UseDoublingAlgorithm,DSGEModel.DAMaximumIterationsValue,DSGEModel.DAToleranceValue,1,DSGEModel.AllowUnitRoot,DSGEModel.StateCovConstValue);
         else;
            [logLikeValue,stat,lnLt,Yhat,MSEY,Ksitt1,Ptt1] = KalmanFilter(DSGEModel.Y(:,FirstPeriod:LastPeriod),[],A,H,F,B0*B0',R,DSGEModel.InitialStateVector,DSGEModel.UseDoublingAlgorithm,DSGEModel.DAMaximumIterationsValue,DSGEModel.DAToleranceValue,1,DSGEModel.AllowUnitRoot,DSGEModel.StateCovConstValue);
         end;
      else;
         if isempty(DSGEModel.X)==0;
            [logLikeValue,stat,lnLt,Yhat,MSEY,Ksitt1,Ptt1] = UnitRootKalmanFilter(DSGEModel.Y(:,FirstPeriod:LastPeriod),DSGEModel.X(:,FirstPeriod:LastPeriod),A,H,F,B0*B0',R,DSGEModel.InitialStateVector,DSGEModel.UseDoublingAlgorithm,DSGEModel.DAMaximumIterationsValue,DSGEModel.DAToleranceValue,1,DSGEModel.AllowUnitRoot,DSGEModel.StateCovConstValue,GetPosition(DSGEModel.StateVariablePositions,DSGEModel.UnitRootStates));
         else;
            [logLikeValue,stat,lnLt,Yhat,MSEY,Ksitt1,Ptt1] = UnitRootKalmanFilter(DSGEModel.Y(:,FirstPeriod:LastPeriod),[],A,H,F,B0*B0',R,DSGEModel.InitialStateVector,DSGEModel.UseDoublingAlgorithm,DSGEModel.DAMaximumIterationsValue,DSGEModel.DAToleranceValue,1,DSGEModel.AllowUnitRoot,DSGEModel.StateCovConstValue,GetPosition(DSGEModel.StateVariablePositions,DSGEModel.UnitRootStates));
         end;
      end;
      %
      % calculate the smooth estimates
      %
      [Ksitt,Ptt,KsitT,PtT,Ksit1t] = StateSmoother(DSGEModel.Y(:,FirstPeriod:LastPeriod),Yhat,Ksitt1,Ptt1,H,F,R);
      %
      % estimate the economic shocks
      %
      etatT = inv(B0'*B0)*B0'*(KsitT(:,2:T)-(F*KsitT(:,1:T-1)));
      %
      % as the first observation we assume that xi0T = 0
      %
      etatT = [inv(B0'*B0)*B0'*KsitT(:,1) etatT];
      Ksitt = Ksitt(:,DSGEModel.KalmanFirstObservation:T);
      KsitT = KsitT(:,DSGEModel.KalmanFirstObservation:T);
      etatT = etatT(:,DSGEModel.KalmanFirstObservation:T);
      if rank(R)>0;
         if isempty(X)==0;
            wtT = Y-(A'*X)-(H'*KsitT);
         else;
            wtT = Y-(H'*KsitT);
         end;
      else;
         wtT = zeros(n,Ttot);
      end;
      %
      % we can now perform the decomposition
      %
      HFED(LastDraw).HistDecompStructure = DSGEHistDecompFcn(Y,X,A,H,F,B0,Ksitt,KsitT,etatT,wtT,h);
      %
      MeanEstimationTime = (((it-1)/it)*MeanEstimationTime)+((1/it)*toc);
      %
      % check if we should save to disk
      %
      if (LastDraw==SaveAfterDraws)|(it==NumDraws);
         ThisSave = ThisSave+1;
         LastDraw = 0;
         %
         % set up the file name and then save
         %
         file = [DSGEModel.OutputDirectory '\hfed\DSGE-HFED-' int2str(h) '-' DSGEModel.NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(ThisSave,'%0.0f') '-' num2str(NumFiles,'%0.0f') '.' num2str(NumDraws,'%0.0f') '-' num2str(TotalDraws,'%0.0f') '.mat'];
         save(file,'HFED','DateVector','KeepShocks','Rmode');
         %
         % restore the data structure
         %
         HFED = [];
      end;
      drawnow;
   end;
else;
   for it=1:NumDraws;
      LastDraw = LastDraw+1;
      if DSGEModel.ShowProgress==1;
         abort = get(WaitHandle,'UserData');
         if strcmp(abort,'cancel')==1;
            break;
         else;
            ProgressDLG([it/NumDraws MeanEstimationTime]);
         end;
      end;
      %
      % Solve the model for the current parameter vector
      %
      theta = thetaPostSample(it,:)';
      tic;
      [A,H,R,F,B0,status] = SolveDSGEModel(DSGEModel,theta,thetaPositions,ModelParameters,AIMData);
      if status~=1;
         %
         % use posterior mode estimate as a backup
         %
         [A,H,R,F,B0,status] = SolveDSGEModel(DSGEModel,thetaMode,thetaPositions,ModelParameters,AIMData);
      end;
      B0 = B0(:,KeepShocks);
      %
      % run the Kalman filter to get the estimate of the state forecasts
      %
      if isempty(DSGEModel.UnitRootStates)==1;
         if isempty(DSGEModel.X)==0;
            [logLikeValue,stat,lnLt,Yhat,MSEY,Ksitt1,Ptt1] = KalmanFilterHt(DSGEModel.Y(:,FirstPeriod:LastPeriod),DSGEModel.X(:,FirstPeriod:LastPeriod),A,H(:,:,FirstPeriod:LastPeriod),F,B0*B0',R,DSGEModel.InitialStateVector,DSGEModel.UseDoublingAlgorithm,DSGEModel.DAMaximumIterationsValue,DSGEModel.DAToleranceValue,1,DSGEModel.AllowUnitRoot,DSGEModel.StateCovConstValue);
         else;
            [logLikeValue,stat,lnLt,Yhat,MSEY,Ksitt1,Ptt1] = KalmanFilterHt(DSGEModel.Y(:,FirstPeriod:LastPeriod),[],A,H(:,:,FirstPeriod:LastPeriod),F,B0*B0',R,DSGEModel.InitialStateVector,DSGEModel.UseDoublingAlgorithm,DSGEModel.DAMaximumIterationsValue,DSGEModel.DAToleranceValue,1,DSGEModel.AllowUnitRoot,DSGEModel.StateCovConstValue);
         end;
      else;
         if isempty(DSGEModel.X)==0;
            [logLikeValue,stat,lnLt,Yhat,MSEY,Ksitt1,Ptt1] = UnitRootKalmanFilterHt(DSGEModel.Y(:,FirstPeriod:LastPeriod),DSGEModel.X(:,FirstPeriod:LastPeriod),A,H(:,:,FirstPeriod:LastPeriod),F,B0*B0',R,DSGEModel.InitialStateVector,DSGEModel.UseDoublingAlgorithm,DSGEModel.DAMaximumIterationsValue,DSGEModel.DAToleranceValue,1,DSGEModel.AllowUnitRoot,DSGEModel.StateCovConstValue,GetPosition(DSGEModel.StateVariablePositions,DSGEModel.UnitRootStates));
         else;
            [logLikeValue,stat,lnLt,Yhat,MSEY,Ksitt1,Ptt1] = UnitRootKalmanFilterHt(DSGEModel.Y(:,FirstPeriod:LastPeriod),[],A,H(:,:,FirstPeriod:LastPeriod),F,B0*B0',R,DSGEModel.InitialStateVector,DSGEModel.UseDoublingAlgorithm,DSGEModel.DAMaximumIterationsValue,DSGEModel.DAToleranceValue,1,DSGEModel.AllowUnitRoot,DSGEModel.StateCovConstValue,GetPosition(DSGEModel.StateVariablePositions,DSGEModel.UnitRootStates));
         end;
      end;
      %
      % calculate the smooth estimates
      %
      [Ksitt,Ptt,KsitT,PtT,Ksit1t] = StateSmootherHt(DSGEModel.Y(:,FirstPeriod:LastPeriod),Yhat,Ksitt1,Ptt1,H(:,:,FirstPeriod:LastPeriod),F,R);
      %
      % estimate the economic shocks
      %
      etatT = inv(B0'*B0)*B0'*(KsitT(:,2:T)-(F*KsitT(:,1:T-1)));
      %
      % as the first observation we assume that xi0T = 0
      %
      etatT = [inv(B0'*B0)*B0'*KsitT(:,1) etatT];
      Ksitt = Ksitt(:,DSGEModel.KalmanFirstObservation:T);
      KsitT = KsitT(:,DSGEModel.KalmanFirstObservation:T);
      etatT = etatT(:,DSGEModel.KalmanFirstObservation:T);
      wtT = zeros(n,Ttot);
      if rank(R)>0;
         if isempty(X)==0;
            for t=1:size(Y,2);
               wtT(:,t) = Y(:,t)-(A'*X(:,t))-(H(:,:,FirstPeriod+DSGEModel.KalmanFirstObservation-2+t)'*KsitT(:,t));
            end;
         else;
            for t=1:size(Y,2);
               wtT(:,t) = Y(:,t)-(H(:,:,FirstPeriod+DSGEModel.KalmanFirstObservation-2+t)'*KsitT(:,t));
            end;
         end;
      end;
      %
      % we can now perform the decomposition
      %
      HFED(LastDraw).HistDecompStructure = DSGEHistDecompFcn(Y,X,A,H(:,:,FirstPeriod+DSGEModel.KalmanFirstObservation-1:LastPeriod),F,B0,Ksitt,KsitT,etatT,wtT,h);
      %
      MeanEstimationTime = (((it-1)/it)*MeanEstimationTime)+((1/it)*toc);
      %
      % check if we should save to disk
      %
      if (LastDraw==SaveAfterDraws)|(it==NumDraws);
         ThisSave = ThisSave+1;
         LastDraw = 0;
         %
         % set up the file name and then save
         %
         file = [DSGEModel.OutputDirectory '\hfed\DSGE-HFED-' int2str(h) '-' DSGEModel.NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(ThisSave,'%0.0f') '-' num2str(NumFiles,'%0.0f') '.' num2str(NumDraws,'%0.0f') '-' num2str(TotalDraws,'%0.0f') '.mat'];
         save(file,'HFED','DateVector','KeepShocks','Rmode');
         %
         % restore the data structure
         %
         HFED = [];
      end;
      drawnow;
   end;
end;
%
% close the wait dialog
%
if DSGEModel.ShowProgress==1;
   if ishandle(WaitHandle)==1;
      set(WaitHandle,'UserData','done');
      close(WaitHandle);
      drawnow;
      pause(0.02);
   end;
else;
   set(WaitControls.text,'UserData','done');
   delete(WaitHandle);
   drawnow;
   pause(0.02);
end;
%
% check if we have computed all output
%
if strcmp(abort,'cancel')==0;
   DoneCalc = 1;
end;

%
% end of DSGEForecastErrorDecomposition.m
%
