function [FEVDs,ConvValue,status] = DSGECondVarianceDecompostion(theta,thetaPositions,ModelParameters,DSGEModel,CurrINI,UseLevels)
% DSGECondVarianceDecompostion: Computes the conditional forecast error variance decompositions for observed variables for a given
%                               value of the parameters.
%
% USAGE:
%
%       [FEVDs,ConvValue,status] = DSGECondVarianceDecompostion(theta,thetaPositions,ModelParameters,DSGEModel,CurrINI,UseLevels)
%
% REQUIRED INPUT:  theta (vector) with values of the original DSGE model parameters
%
%                  thetaPositions (structure) with fields parameter and distribution. This function
%                                 needs the former field to know which parameter name corresponds to
%                                 position i in theta.
%
%                  ModelParameters (structure) whose fields have names given by the parameter names in the
%                                 DSGE model.
%
%                  DSGEModel (structure) with fields containing all the information about a DSGE Model for YADA.
%
%                  CurrINI (structure) with initialization information.
%
%                  UseLevels (interger) which indicates if variables in first differences should be
%                                 accumulated (1), annualized (2), or left alone (0).
%
% REQUIRED OUTPUT: FEVDs (nxqxh) matrix with the forecast error variance decompositions. For each h, the nxq matrix
%                                 is ordered such the the q economic are in the columns and the n observed variables in
%                                 the rows.
%
%                  ConvValue (nx1) vector with the ratio between the conditional forecast error variance at period h
%                                 and the estimate of the long-run forecast error variance. This output is non-empty
%                                 only when UseLevels is 1.
%
% OPTIONAL OUTPUT: status (integer) Indicates whether the solution of the model is unique or not.
%                                 mcode is 1 when unique, 0 if not computed, and various other values when
%                                 there are problems with the solution. Specifically:
%
%                                      mcode = 2:      Roots not correctly computed by real_schur
%                                      mcode = 3:      Too many big roots
%                                      mcode = 35:     Too many big roots, and q(:,right) is singular
%                                      mcode = 4:      Too few big roots
%                                      mcode = 45:     Too few big roots, and q(:,right) is singular
%                                      mcode = 5:      q(:,right) is singular
%                                      mcode = 61:     Too many exact shiftrights
%                                      mcode = 62:     Too many numeric shiftrights
%                                      mcode = 7:      The a matrix has infinite or NaN values. Eigenvalues
%                                                      cannot be calculated.
%                                      mcode = 8:      The function "compute_aim_matrices" returns complex
%                                                      numbers.
%                                      else            Return code not properly specified
%
% NOTE: Two output arguments should only be used when theta has the initial values.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: December 4, 2006.
%                        This version: March 9, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 06-12-2006: Removed the input R, the covariance matrix of the measurement errors, in the call to
%               the function "DSGECondVarianceDecompFcn".
%
% * 07-12-2006: Took the change from CurrINI.IRHorizon to DSGEModel.IRHorizon into account.
%
% * 08-12-2006: Added the UseLevels input variable and fixed the code to account for it.
%
% * 23-01-2007: The "UseLevels" variable can now take on the value 2, implying that annualized conditional
%               variance decompositions are calculated.
%
% * 25-01-2007: Fixed a bug when supplying the function "DSGECondAnnVarianceDecompFcn" with input based
%               on the "DSGEModel.annual" vector.
%
% * 13-11-2007: Updated the documentation.
%
% * 14-04-2008: Added the output argument "ConvValue".
%
% * 16-04-2008: The code now checks if the model has a unique convergent solution also at the posterior mode
%               estimates.
%
% * 16-05-2008: Took the "DSGEModel.AIMTolerance" input for AiMSolver into account.
%
% * 23-05-2008: Updated the documentation.
%
% * 27-05-2008: The mcode value 7 is now supported.
%
% * 03-07-2008: Made sure that the measurement equation receives only the active state variable names.
%               Added the try-catch code for AiMSolver.
%
% * 09-07-2008: Added code needed to handle a time-varying measurement matrix H.
%
% * 15-07-2008: Fixed a bug where "SelPos" was not applied to a 3D H-matrix.
%
% * 23-01-2009: Updated the documentation.
%
% * 30-01-2009: Made sure that status is only set to mcode when the latter exists.
%
% * 09-02-2009: Changed the input "length(DSGEModel.StateVariablePositions)" to "size(F,1)" when calling the
%               measurement equation function.
%
% * 13-02-2009: The mcode values 7 and 8 are now properly mentioned when encountered.
%
% * 09-03-2009: Added quotation marks around paths for the dos copy commands.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize the output
%
FEVDs = [];
ConvValue = [];
%
% set up the ModelParameters structure based on theta and thetaPositions
%
if nargout==2;
   %
   % Make sure that the ModelParameter structure takes the correct values.
   % This is not necessary to do when initial values are given to the function.
   % In that case, more substantial checking is required!
   %
   ModelParameters = ThetaToModelParameters(ModelParameters,theta,thetaPositions);
   %
   % copy files to the tmp folder
   %
   [stat,msg] = dos(['copy /Y "' GetPath(DSGEModel.AIMDataFile) 'compute_aim_matrices.m" "' pwd '\tmp"']);
   [stat,msg] = dos(['copy /Y "' DSGEModel.MeasurementEquationFile '" "' pwd '\tmp"']);
   if FileExist(DSGEModel.UpdateParameterFile)==1;
      [stat,msg] = dos(['copy /Y "' DSGEModel.UpdateParameterFile '" "' pwd '\tmp"']);
   end;
   %
   % Update any parameteters that need updating
   %
   if FileExist(DSGEModel.UpdateParameterFile)==1;
      eval(['ModelParameters = ' GetName(DSGEModel.UpdateParameterFile) '(ModelParameters);']);
   end;
end;
%
% load the AIMData file
%
ErrorStr = '';
try;
   AIMData = load(DSGEModel.AIMDataFile);
catch;
   ErrorStr = ['Unable to load the file "' DSGEModel.AIMDataFile '". Message caught is: ' lasterr];
end;
if isempty(ErrorStr)==1;
   %
   % See if we can solve the model
   %
   if nargout==2;
      [mcode,MaxAbsError,BMatrix,SMatrix] = AiMSolver(ModelParameters,AIMData.neq,AIMData.nlag,AIMData.nlead,DSGEModel.AIMTolerance);
   else;
      try;
         [mcode,MaxAbsError,BMatrix,SMatrix] = AiMSolver(ModelParameters,AIMData.neq,AIMData.nlag,AIMData.nlead,DSGEModel.AIMTolerance);
      catch;
         if exist('mcode')>0;
            status = mcode;
         else;
            status = 0;
         end;
         ErrorStr = ['There was an error running AiM. YADA received the following message: ' lasterr];
         About(ErrorStr,'error','AiM Error',300,500,CurrINI);
         drawnow;
         return;
      end;
   end;
   if nargout==3;
      status = mcode;
      if status~=1;
         return;
      end;
   else;
      if mcode~=1;
         if mcode==2;
            mcodeStr = 'Roots are not correctly computed by real_schur.'
         elseif mcode==3;
            mcodeStr = 'Too many big roots.';
         elseif mcode==35;
            mcodeStr = 'Too many big roots, and q(:,right) is singular.';
         elseif mcode==4;
            mcodeStr = 'Too few big roots.';
         elseif mcode==45;
            mcodeStr = 'Too few big roots, and q(:,right) is singular.';
         elseif mcode==5;
            mcodeStr = 'q(:,right) is singular.';
         elseif mcode==61;
            mcodeStr = 'Too many exact shiftrights.';
         elseif mcode==62;
            mcodeStr = 'Too many numeric shiftrights.';
         elseif mcode==7;
            mcodeStr = 'Infinite or NaN values detected.';
         elseif mcode==8;
            mcodeStr = 'The function "compute_aim_matrices" returns complex numbers.';
         else;
            mcodeStr = 'Return code not properly specified.';
         end;
         txt = ['The AiM solver provided the return code: ' num2str(mcode) ', i.e., "' mcodeStr '"'];
         About(txt,'information','YADA - AiM Solver Error',200,500,CurrINI);
         return;
      end;
   end;
   %
   % Convert to the state equation matrices
   %
   [F,B0] = AiMtoStateSpace(BMatrix,SMatrix,DSGEModel.StateVariablePositions,DSGEModel.StateShockPositions);
   %
   % obtain the measurement equation matrices
   %
   try;
      eval(['[A,H,R] = ' GetName(DSGEModel.MeasurementEquationFile) '(ModelParameters,AIMData.endog(DSGEModel.StateVariablePositions,:),DSGEModel.VariableNames,DSGEModel.XVariableNames,DSGEModel.n,size(F,1),DSGEModel.k);']);
   catch;
      %
      % display an error message
      %
      if nargout==3;
         status = 0;
      end;
      ErrorStr = ['Error found in "' GetName(DSGEModel.MeasurementEquationFile) '.m". Message caught is: ' lasterr];
      About(ErrorStr,'error','Error - Measurement Equation',150,500,CurrINI);
      return;
   end;
   if length(size(H))==3;
      %
      % check if H has the required size
      %
      [FirstPeriod,LastPeriod] = CreateSubSample(DSGEModel);
      if size(H,3)>=LastPeriod;
         %
         % pick a certain period for the H matrix
         %
         if nargout==2;
            ParamStr = 'Posterior Mode';
         else;
            ParamStr = 'Initial Values';
         end;
         YADAHandle = findobj('Type','figure','Tag','YADA');
         controls = get(YADAHandle,'UserData');
         SubSampleMat = get(controls.dsge.initializekalman,'String');
         SubSampleMat = SubSampleMat(DSGEModel.KalmanFirstObservation:size(SubSampleMat,1),:);
         [action,SelectedPosition] = SelectionDlg(SubSampleMat,size(SubSampleMat,1),'Select time period for the measurement matrix H','Measurement Matrix',['Conditional Variance Decompositions - ' ParamStr],'','',CurrINI);
         %
         SelPos = FirstPeriod+DSGEModel.KalmanFirstObservation-1+(SelectedPosition-1);
         if strcmp(lower(action),'cancel')==1;
            return;
         end;
         H = H(:,:,SelPos);
      else;
         ErrorStr = ['The number of time periods for the time-varying measurement matrix H (' int2str(size(H,3)) ') is lower than the number of observations (T = ' int2str(LastPeriod) '). YADA has therefore aborted from computing the conditional variance decompositions.'];
         About(ErrorStr,'error','Error - Measurement Equation',200,500,CurrINI);
         return;
      end;
   end;
   %
   % Determine the impulse response horizon
   %
   if strcmp(DSGEModel.DataFrequency,'q')==1;
      SampleFreq = 4;
   elseif strcmp(DSGEModel.DataFrequency,'m')==1;
      SampleFreq = 12;
   else;
      SampleFreq = 1;
   end;
   IRHorizon = SampleFreq*DSGEModel.IRHorizon;
   %
   % Now we can compute the variance decompositions
   %
   if UseLevels==0;
      FEVDs = DSGECondVarianceDecompFcn(H,F,B0,IRHorizon);
   elseif UseLevels==1;
      [FEVDs,ConvValue] = DSGECondLevVarianceDecompFcn(H,F,B0,diag(1-DSGEModel.levels),IRHorizon);
   else;
      if strcmp(DSGEModel.DataFrequency,'q')==1;
         SampleFreq = 4;
      elseif strcmp(DSGEModel.DataFrequency,'m')==1;
         SampleFreq = 12;
      else;
         return;
      end;
      FEVDs = DSGECondAnnVarianceDecompFcn(H,F,B0,diag(DSGEModel.annual==SampleFreq),IRHorizon,SampleFreq);
   end;
else;
   if nargout==3;
      status = 0;
   end;
   About(ErrorStr,'information','YADA - Bad MAT File',120,500,CurrINI);
end;

%
% end of DSGECondVarianceDecompostion.m
%
