function [FEVDs,ConvValue] = DSGECondLevVarianceDecompFcn(H,F,B0,AccMat,h)
% DSGECondLevVarianceDecompFcn: Calculates conditional the forecast error variance decompositions for
%                               the levels data.
%
% USAGE:
%
%       [FEVDs,ConvValue] = DSGECondVarianceDecompFcn(H,F,B0,AccMat,h)
%
% REQUIRED INPUT:  H (rxn) matrix where n is the number of observed variables and r the number of
%                          state variables. This matrix is part of the measurement equation of the
%                          state-space representation for the Kalman filter.
%
%                  F (rxr) matrix from the state equation.
%
%                  B0 (rxq) matrix, where B0*B0' is the covariance matrix for the shocks to the
%                          state equation.
%
%                  AccMat (nxn) diagonal 0-1 matrix. Only those diagonal elements that that 1
%                          will be accumulated in the
%
%                  h (positive integer) giving the forecast horizon.
%
% REQUIRED OUTPUT: FEVDs (nxqxh) matrix with the forecast error variance decompositions. For each h, the nxq matrix is
%                          ordered such the the q economic are in the columns and the n observed variables in the rows.
%
% OPTIONAL OUTPUT: ConvValue (nx1) vector with the ratio between the conditional forecast error variance at period h
%                          and the estimate of the long-run forecast error variance.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: December 8, 2006.
%                        This version: January 23, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 11-12-2006: Updated the documentation.
%
% * 13-11-2007: Updated the documentation.
%
% * 14-04-2008: Made sure that small responses are set to 0. Made sure that an estimate of the long-run variance
%               decomposition is provided. The optional output variable "ConvValue" has also been added.
%
% * 23-05-2008: Updated the documentation.
%
% * 23-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[r n] = size(H);
q = size(B0,2);
FEVDs = zeros(n,q,h+1);
if nargout==2;
   ConvValue = [];
end;
VarShares = zeros(n,1);
DenomOld = zeros(n,n);
NumerOld = zeros(n,q);
RiOld = zeros(n,q);
Fi = eye(r);
%
% set small responses to zero
%
Bpos = (B0>1.0e-12);
Bneg = (B0<1.0e-12);
B0 = (B0 .* Bpos)+(B0 .* Bneg);
for i=1:h;
   Ri = RiOld+(H'*Fi*B0);
   Ripos = (Ri>1.0e-12);
   Rineg = (Ri<1.0e-12);
   Ri = (Ri .* Ripos)+(Ri .* Rineg);
   %
   RiOld = AccMat*Ri;
   DenomOld = DenomOld+((Ri*Ri') .* eye(n));
   NumerOld = NumerOld+(Ri .* Ri);
   FEVDs(:,:,i) = inv(DenomOld)*NumerOld;
   Fi = Fi*F;
end;
VarShares = diag((AccMat*DenomOld/h)+((eye(n)-AccMat)*DenomOld));
%
% estimate the long-run
%
if h<200;
   for i=h+1:200;
      Ri = RiOld+(H'*Fi*B0);
      Ripos = (Ri>1.0e-12);
      Rineg = (Ri<1.0e-12);
      Ri = (Ri .* Ripos)+(Ri .* Rineg);
      %
      RiOld = AccMat*Ri;
      DenomOld = DenomOld+((Ri*Ri') .* eye(n));
      NumerOld = NumerOld+(Ri .* Ri);
      Fi = Fi*F;
   end;
end;
DenomNew = (AccMat*((RiOld*RiOld') .* eye(n)))+((eye(n)-AccMat)*DenomOld);
NumerNew = (AccMat*(RiOld .* RiOld))+((eye(n)-AccMat)*NumerOld);
FEVDs(:,:,h+1) = inv(DenomNew)*NumerNew;
VarShares = VarShares./diag(DenomNew);
if nargout==2;
   ConvValue = VarShares;
end;

%
% end of DSGECondLevVarianceDecompFcn.m
%
