function CorrData = DSGECondCorrDistribution(DSGEModel,CurrINI,SelectedParameters,NumDraws,TotalDraws,CurrChain,NumPaths)
% DSGECondCorrDistribution: Computes the mean and the quantiles of the conditional correlations
%
% USAGE:
%
%       CorrData = DSGECondCorrDistribution(DSGEModel,CurrINI,SelectedParameters,NumDraws,TotalDraws,CurrChain,NumPaths)
%
% REQUIRED INPUT:  DSGEModel (structure) with fields containing all the information about a DSGE Model for YADA.
%
%                  CurrINI (structure) with initialization information.
%
%                  SelectedParameters (vector) of dimension NumParam which indicates which parameters can vary from
%                                     one draw to the next (1) and which parameters are fixed at posterior mode (0).
%
%                  NumDraws (integer) with the number of effective draws from the posterior distribution of the
%                                 parameters.
%
%                  TotalDraws (integer) with the number of draws from the posterior distribution of the parameters
%                                 that could have been used.
%
%                  CurrChain (integer) with the number for the current Markov chain.
%
%                  NumPaths (integer) that is positive if results for simulated data should be checked and
%                                 0 otherwise.
%
% REQUIRED OUTPUT: CorrData (structure) with fields Mean and Quantiles and ShockNames. The length of
%                                 CorrData is equal to the number of shocks.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: February 17, 2009.
%                        This version: February 19, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 18-02-2009: Continued work on the function.
%
% * 19-02-2009: Finished the function with the code for the quantiles output.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
CorrData = [];
%
% fix the information needed to setup the file names
%
SelParamStr = strrep(num2str(SelectedParameters),' ','');
%
if NumPaths==0;
   NumFiles = ceil(NumDraws/min(NumDraws,1000));
   TotalPaths = NumDraws;
else;
   NumFiles = ceil(NumDraws/1);
   TotalPaths = NumDraws*NumPaths;
end;
NumQuants = length(DSGEModel.Percentiles);
%
% we first need to determine which shocks have been used.
%
if NumPaths==0;
   file = [DSGEModel.OutputDirectory '\condcorr\DSGE-CondCorr-' DSGEModel.NameOfModel '-' num2str(CurrChain,'%0.0f') '-' SelParamStr '-1-' num2str(NumFiles,'%0.0f') '.' num2str(NumDraws,'%0.0f') '-' num2str(TotalDraws,'%0.0f') '.mat'];
   SampleTypeStr = 'Population';
else;
   file = [DSGEModel.OutputDirectory '\condcorr\DSGE-CondCorr-' DSGEModel.NameOfModel '-' num2str(CurrChain,'%0.0f') '-' SelParamStr '-1-' num2str(NumFiles,'%0.0f') '.' num2str(NumDraws,'%0.0f') '-' num2str(TotalDraws,'%0.0f') '-' num2str(NumPaths,'%0.0f') '.mat'];
   SampleTypeStr = 'Sample';
end;
%
MatData = load(file);
KeepShocks = MatData.KeepShocks;
q = length(KeepShocks);
n = DSGEModel.n;
CorrData(1).ShockNames = DSGEModel.StateShockNames(KeepShocks,:);
Dn = DuplicationMatrix(n);
%
if (NumPaths>0)|(TotalPaths>=1000);
   txt = ['Please wait while YADA computes the distributions for the model based ' lower(SampleTypeStr) ' conditional correlations of the observed variables. Current shock: ' StringTrim(CorrData(1).ShockNames(1,:)) '.'];
   WaitHandle = WaitDLG(txt,'information',['Conditional Correlations - ' SampleTypeStr '-Based'],500,200,CurrINI,1);
   WaitControls = get(WaitHandle,'UserData');
end;
%
% we should probably shown a wait dialog
%
for j=1:q;
   if (NumPaths>0)|(TotalPaths>=1000);
      txt = ['Please wait while YADA computes the distributions for the model based ' lower(SampleTypeStr) ' conditional correlations of the observed variables. Current shock: ' StringTrim(CorrData(1).ShockNames(j,:)) '.'];
      set(WaitControls.text,'String',txt);
      drawnow;
      pause(0.02);
   end;
   %
   %
   % initialize output
   %
   CorrData(j).Mean = zeros(n,n);
   TmpCorr = zeros(TotalPaths,(n*(n+1))/2);
   FirstDraw = 1;
   LastDraw = 0;
   for ThisSave=1:NumFiles;
      if NumPaths==0;
         file = [DSGEModel.OutputDirectory '\condcorr\DSGE-CondCorr-' DSGEModel.NameOfModel '-' num2str(CurrChain,'%0.0f') '-' SelParamStr '-' num2str(ThisSave,'%0.0f') '-' num2str(NumFiles,'%0.0f') '.' num2str(NumDraws,'%0.0f') '-' num2str(TotalDraws,'%0.0f') '.mat'];
      else;
         file = [DSGEModel.OutputDirectory '\condcorr\DSGE-CondCorr-' DSGEModel.NameOfModel '-' num2str(CurrChain,'%0.0f') '-' SelParamStr '-' num2str(ThisSave,'%0.0f') '-' num2str(NumFiles,'%0.0f') '.' num2str(NumDraws,'%0.0f') '-' num2str(TotalDraws,'%0.0f') '-' num2str(NumPaths,'%0.0f') '.mat'];
      end;
      MatData = load(file);
      DrawSize = length(MatData.CC);
      %
      % compute the mean
      %
      for p=1:DrawSize;
         CorrData(j).Mean = CorrData(j).Mean+((1/TotalPaths)*MatData.CC(p).Shock(j).CondCorr);
      end;
      %
      % now the quantile data for the correlations
      %
      LastDraw = LastDraw+DrawSize;
      for p=FirstDraw:LastDraw;
         TmpCorr(p,:) = vech(MatData.CC(p-FirstDraw+1).Shock(j).CondCorr)';
      end;
      FirstDraw = FirstDraw+DrawSize;
   end;
   %
   % sort all columns in TmpCorr
   %
   for i=1:size(TmpCorr,2);
      TmpCorr(:,i) = sort(TmpCorr(:,i));
   end;
   %
   % now we can determine the quantiles
   %
   for i=1:NumQuants;
      CorrData(j).Quantiles(i).percent = DSGEModel.Percentiles(i);
      if DSGEModel.Percentiles(i)<50;
         QuantVal = ceil((DSGEModel.Percentiles(i)/100)*TotalPaths);
      else;
         QuantVal = floor((DSGEModel.Percentiles(i)/100)*TotalPaths);
      end;
      CorrData(j).Quantiles(i).Mean = reshape(Dn*TmpCorr(QuantVal,:)',n,n);
   end;
end;
%
% close the wait dialog
%
if (NumPaths>0)|(TotalPaths>=1000);
   set(WaitControls.text,'UserData','done');
   delete(WaitHandle);
   drawnow;
   pause(0.02);
end;

%
% end of DSGECondCorrDistribution.m
%
