function FEVDs = DSGECondAnnVarianceDecompFcn(H,F,B0,AccMat,h,AccHorizon)
% DSGECondAnnVarianceDecompFcn: Calculates conditional the forecast error variance decompositions for
%                               the annualized data.
%
% USAGE:
%
%       FEVDs = DSGECondAnnVarianceDecompFcn(H,F,B0,AccMat,h,AccHorizon)
%
% REQUIRED INPUT:  H (rxn) matrix where n is the number of observed variables and r the number of
%                          state variables. This matrix is part of the measurement equation of the
%                          state-space representation for the Kalman filter.
%
%                  F (rxr) matrix from the state equation.
%
%                  B0 (rxq) matrix, where B0*B0' is the covariance matrix for the shocks to the
%                          state equation.
%
%                  AccMat (nxn) diagonal 0-1 matrix. Only those diagonal elements that that 1
%                          will be accumulated in the
%
%                  h (positive integer) giving the forecast horizon.
%
%                  AccHorizon (integer) giving the annualization rate; quarterly = 4, monthly = 12.
%
% REQUIRED OUTPUT: FEVDs (nxqxh) matrix with the forecast error variance decompositions. For each h, the nxq matrix is
%                          ordered such the the q economic are in the columns and the n observed variables in the rows.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: January 23, 2007.
%                        This version: January 23, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-11-2007: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 23-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[r n] = size(H);
q = size(B0,2);
FEVDs = zeros(n,q,h);
DenomOld = zeros(n,n);
NumerOld = zeros(n,q);
SaveAccMat = zeros(n,q,h);
Fi = eye(r);
for i=1:h;
   SaveAccMat(:,:,i) = (H'*Fi*B0);
   %
   % re-initialize
   %
   RiOld = zeros(n,q);
   for j=max(i+1-AccHorizon,1):i-1;
      RiOld = RiOld+SaveAccMat(:,:,j);
   end;
   Ri = (AccMat*RiOld)+SaveAccMat(:,:,i);
   DenomOld = DenomOld+((Ri*Ri') .* eye(n));
   NumerOld = NumerOld+(Ri .* Ri);
   FEVDs(:,:,i) = inv(DenomOld)*NumerOld;
   Fi = Fi*F;
end;

%
% end of DSGECondAnnVarianceDecompFcn.m
%
