function PredEventData = CalculatePredictionEvents(PredPaths,PredictionEvent)
% CalculatePredictionEvents: Computes the frequency with which the prediction evenst are true
%                            for the prediction paths.
%
% USAGE:
%
%       PredEventData = CalculatePredictionEvents(PredPaths,PredictionEvent)
%
% REQUIRED INPUT:  PredPaths (n x h x NumPaths) 3D matrix with the NumPaths possible paths for the n
%                            endogenous variables.
%
%                  PredictionEvent (n x 3) matrix with the prediction event information. The first
%                            column holds the upper bound, the second the lower bound for the event,
%                            while the third holds the event length in periods.
%
% REQUIRED OUTPUT: PredEventData (n x 3) matrix where the first column holds the number of times
%                            the PredPaths data satisfies the event, the second hold the number
%                            of times the PredPaths data is below the lower bound of the event,
%                            while the third column holds the number of times the event has been
%                            tested.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: May 3, 2007.
%                        This version: January 22, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 04-05-2007: The function now tests if the upper bound is satisfied as well as the lower bound.
%
% * 06-06-2007: The PredEventData now has the second column function. Previously, only the first
%               and the "third" were included.
%
% * 13-11-2007: Updated the documentation.
%
% * 04-04-2008: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
[n,h,NumPaths] = size(PredPaths);
PredEventData = zeros(n,3);
%
% we check the events variable by variable
%
for i=1:n;
   EventDataUpper = PredictionEvent(i,1)*ones(1,round(PredictionEvent(i,3)));
   EventDataLower = PredictionEvent(i,2)*ones(1,round(PredictionEvent(i,3)));
   EventLength = round(PredictionEvent(i,3));
   NumTests = h-EventLength+1;
   for p=1:NumPaths;
      for j=1:NumTests;
         PredData = PredPaths(i,j:EventLength+j-1,p);
         PredEventData(i,1) = PredEventData(i,1)+min(min(PredData>=EventDataLower),min(PredData<=EventDataUpper));
         PredEventData(i,2) = PredEventData(i,2)+min(PredData<EventDataLower);
      end;
   end;
   PredEventData(i,3) = NumTests*NumPaths;
end;

%
% end of CalculatePredictionEvents.m
%
