function PredictionErrors = CalculatePredictionErrors(MeanPrediction,AnnualizeData,TransData,ModelType,DSGEModel)
% CalculatePredictionErrors: Computes the prediction errors using a point forecast.
%
% USAGE:
%
%       PredictionErrors = CalculatePredictionErrors(MeanPrediction,AnnualizeData,TransData,ModelType,DSGEModel)
%
% REQUIRED INPUT:  MeanPrediction (n x h) matrix with the point forecasts.
%
%                  AnnualizeData (boolean) that is 1 if the prediction paths should be annualized and 0 otherwise.
%
%                  TransData (boolean) that is 1 if the prediction paths should be transformed and 0 otherwise.
%
%                  ModelType (string vector) taking on the values "DSGE" or "BVAR".
%
%                  DSGEModel (structure) with fields containing all the information about a DSGE Model for YADA.
%
% REQUIRED OUTPUT: PredictionErrors (n x h1) matrix with the prediction errors that can be computed. The
%                                   integer h1 <= h. The output is empty if there are no observed variables
%                                   for the prediction sample period.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: May 4, 2007.
%                        This version: January 22, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 06-06-2007: Updated the documentation.
%
% * 13-11-2007: Updated the documentation.
%
% * 05-05-2008: Added the "TransData" input.
%
% * 23-05-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
PredictionErrors = [];
[n h] = size(MeanPrediction);
%
% Determine the sample with actual data
%
DateVector = MakeDateVector(DSGEModel.BeginYear,DSGEModel.BeginPeriod,DSGEModel.EndYear,DSGEModel.EndPeriod,DSGEModel.DataFrequency);
[FirstPeriod,LastPeriod] = CreateSubSample(DSGEModel);
%
% check if the last period for the estimation sample is also
% the last period for the observations
%
T = size(DSGEModel.Y,2);
if LastPeriod==T;
   return;
end;
if AnnualizeData==0;
   if TransData==0;
      Y = DSGEModel.Y(:,LastPeriod+1:min(T,LastPeriod+h));
   else;
      [YTStatus,YData,ErrorStr,HeaderStr] = CalcTransformData(DSGEModel.VariableNames,DSGEModel.Y',DSGEModel.YTransformation);
      Tt = size(YData,1);
      Y = YData(Tt+1-min(h,T-LastPeriod):Tt,:)';
   end;
else;
   %
   % we annualize the data keeping th number of
   %
   Y = DSGEModel.Y;
   Yinit = [];
   if max(DSGEModel.annual)>1;
      Yinit = zeros(size(Y,1),max(DSGEModel.annual)-1);
   end;
   Y = Annualize([Yinit Y],DSGEModel.annual,DSGEModel.annualscale);
   Y = Y(:,LastPeriod+1:min(T,LastPeriod+h));
end;
%
% make sure that the correct observed variables are selected in the event that
% the ModelType variable is BVAR
%
if strcmp(ModelType,'BVAR')==1;
   Y = Y(DSGEModel.BVARY,:);
end;
%
% now we can compute the errors
%
h1 = size(Y,2);
PredictionErrors = Y-MeanPrediction(:,1:h1);

%
% end of CalculatePredictionErrors.m
%
