function [MaxEigenvalue,status] = CalculateDSGEMaximumEigenvalue(theta,thetaPositions,ModelParameters,DSGEModel,CurrINI)
% CalculateDSGEMaximumEigenvalue: Calculates the maximum absolute value of the eigenvalues of the F matrix
%                                 in the state equation.
%
% USAGE:
%
%       [MaxEigenvalue,status] = CalculateDSGEMaximumEigenvalue(theta,thetaPositions,ModelParameters,DSGEModel,CurrINI)
%
% REQUIRED INPUT:  theta (vector) with values of the original DSGE model parameters
%
%                  thetaPositions (structure) with fields parameter and distribution. This function
%                                 needs the former field to know which parameter name corresponds to
%                                 position i in theta.
%
%                  ModelParameters (structure) whose fields have names given by the parameter names in the
%                                 DSGE model.
%
%                  DSGEModel (structure) with fields containing all the information about a DSGE Model for YADA.
%
%                  CurrINI (structure) with initialization information
%
% REQUIRED OUTPUT: MaxEigenvalue (scalar) with the largest absolute value of the eigenvalues of F.
%
% OPTIONAL OUTPUT: status (integer) Indicates whether the solution of the model is unique or not.
%                                   mcode is 1 when unique, 0 if not computed, and various other values when
%                                   there are problems with the solution. Specifically:
%
%                                      mcode = 2:      Roots not correctly computed by real_schur
%                                      mcode = 3:      Too many big roots
%                                      mcode = 35:     Too many big roots, and q(:,right) is singular
%                                      mcode = 4:      Too few big roots
%                                      mcode = 45:     Too few big roots, and q(:,right) is singular
%                                      mcode = 5:      q(:,right) is singular
%                                      mcode = 61:     Too many exact shiftrights
%                                      mcode = 62:     Too many numeric shiftrights
%                                      mcode = 7:      The a matrix has infinite or NaN values. Eigenvalues
%                                                      cannot be calculated.
%                                      mcode = 8:      The function "compute_aim_matrices" returns complex
%                                                      numbers.
%                                      else            Return code not properly specified
%
% NOTE: Two output arguments should only be used when theta has the initial values.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: August 22, 2006.
%                        This version: February 13, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 25-08-2006: Deleted instructions to copy compute_aim_matrices and the file with parameters to update to the
%               tmp directory. These files have already been copied there when the function is called.
%
% * 17-10-2006: Added a try, catch call when attempting to load mat-files. This is needed if matlab can't
%               read the mat-file, e.g., because it was create with a later version of matlab.
%
% * 09-03-2007: Made sure that the missing input "CurrINI" was included.
%
% * 13-11-2007: Updated the documentation.
%
% * 16-04-2008: The code now checks if the model has a unique convergent solution also at the posterior mode
%               estimates.
%
% * 16-05-2008: Took the "DSGEModel.AIMTolerance" input for AiMSolver into account.
%
% * 23-05-2008: Updated the documentation.
%
% * 27-05-2008: The mcode value 7 is now supported.
%
% * 03-07-2008: Added the optional "status" output variable.
%
% * 22-01-2009: Updated the documentation.
%
% * 30-01-2009: Made sure that status is only set to mcode when the latter exists.
%
% * 13-02-2009: The mcode values 7 and 8 are now properly mentioned when encountered.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

MaxEigenvalue = [];
%
% Make sure that the ModelParameter structure takes the correct values.
% This is not necessary to do when initial values are given to the function.
% In that case, more substantial checking is required!
%
ModelParameters = ThetaToModelParameters(ModelParameters,theta,thetaPositions);
%
% Update any parameteters that need updating
%
if FileExist(DSGEModel.UpdateParameterFile)==1;
   eval(['ModelParameters = ' GetName(DSGEModel.UpdateParameterFile) '(ModelParameters);']);
end;
%
% load the AIMData file
%
ErrorStr = '';
try;
   AIMData = load(DSGEModel.AIMDataFile);
catch;
   ErrorStr = ['Unable to load the file "' DSGEModel.AIMDataFile '". Message caught is: ' lasterr];
end;
if isempty(ErrorStr)==1;
   %
   % See if we can solve the model
   %
   if nargout==1;
      [mcode,MaxAbsError,BMatrix,SMatrix] = AiMSolver(ModelParameters,AIMData.neq,AIMData.nlag,AIMData.nlead,DSGEModel.AIMTolerance);
   else;
      try;
         [mcode,MaxAbsError,BMatrix,SMatrix] = AiMSolver(ModelParameters,AIMData.neq,AIMData.nlag,AIMData.nlead,DSGEModel.AIMTolerance);
      catch;
         if exist('mcode')>0;
            status = mcode;
         else;
            status = 0;
         end;
         ErrorStr = ['There was an error running AiM. YADA received the following message: ' lasterr];
         About(ErrorStr,'error','AiM Error',300,500,CurrINI);
         drawnow;
         return;
      end;
   end;
   if nargout>2;
      status = mcode;
      if status~=1;
         return;
      end;
   else;
      if mcode~=1;
         if mcode==2;
            mcodeStr = 'Roots are not correctly computed by real_schur.'
         elseif mcode==3;
            mcodeStr = 'Too many big roots.';
         elseif mcode==35;
            mcodeStr = 'Too many big roots, and q(:,right) is singular.';
         elseif mcode==4;
            mcodeStr = 'Too few big roots.';
         elseif mcode==45;
            mcodeStr = 'Too few big roots, and q(:,right) is singular.';
         elseif mcode==5;
            mcodeStr = 'q(:,right) is singular.';
         elseif mcode==61;
            mcodeStr = 'Too many exact shiftrights.';
         elseif mcode==62;
            mcodeStr = 'Too many numeric shiftrights.';
         elseif mcode==7;
            mcodeStr = 'Infinite or NaN values detected.';
         elseif mcode==8;
            mcodeStr = 'The function "compute_aim_matrices" returns complex numbers.';
         else;
            mcodeStr = 'Return code not properly specified.';
         end;
         txt = ['The AiM solver provided the return code: ' num2str(mcode) ', i.e., "' mcodeStr '"'];
         About(txt,'information','YADA - AiM Solver Error',200,500,CurrINI);
         return;
      end;
   end;
   %
   % Convert to the state equation matrices
   %
   [F,B0] = AiMtoStateSpace(BMatrix,SMatrix,DSGEModel.StateVariablePositions,DSGEModel.StateShockPositions);
   MaxEigenvalue = max(abs(eig(F)));
else;
   if nargout==2;
      status = 0;
   end;
   About(ErrorStr,'information','YADA - Bad MAT File',120,500,CurrINI);
end;

%
% end of CalculateDSGEMaximumEigenvalue.m
%
