function [CUSUMPost,CUSUMAverage] = CUSUM(X,NumBurnin,PlotType,CurrINI)
% CUSUM: Computes the Yu and Mykland (1998) CUSUM paths and a window average
%
% USAGE: [CUSUMPost,CUSUMAverage] = CUSUM(X,NumBurnin,PlotType,CurrINI)
%
% REQUIRED INPUT:  X (matrix) of dimension NumIter x NumStat where NumStat is the number of
%                    statistics to calculate accumulated sums for.
%
%                  NumBurnin (integer) with the number of burn-in draws.
%
%                  PlotType (string vector) that takes on the values "log posterior",
%                    "original parameters", and "transformed parameters".
%
%                  CurrINI (structure) with initialization information.
%
% REQUIRED OUTPUT: CUSUMPost (matrix) of dimension NumIter-NumBurnin x NumStat. The CUSUMPost entries
%                    are calculated relative to the full sample mean, which is then added to
%                    the accumulated difference.
%
%                  CUSUMAverage (matrix) of dimension 0.9*(NumIter-NumBurnin) x NumStat.
%
% REFERENCE:  Yu, B. and Mykland, P. (1998), "Looking at Markov Samplers Through CUSUM
%                       Path Plots: A Simple Diagnostic Idea", Statistics and Computing,
%                       8, 275-286.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: April 18, 2007.
%                        This version: January 22, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 19-04-2007: Added the wait dialog.
%
% * 13-11-2007: Updated the documentation.
%
% * 07-01-2007: Added drawnow calls after Query and About calls.
%
% * 10-01-2008: Changed the last drawnow call to pause(0.02).
%
% * 11-03-2008: Changed the length of the CUSUMAverage matrix so that only NumIter-NumBurnin matter.
%
% * 23-05-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% Initialize
%
CUSUMPost = [];
CUSUMAverage = [];
%
[NumIter NumStat] = size(X);
X = X(NumBurnin+1:NumIter,:);
%
% set up a wait dialog
%
txt = ['Please wait while YADA computes CUSUM estimates of the ' lower(PlotType) '. Estimation started at: ' StringTrim(datestr(now,14)) '.'];
WaitHandle = WaitDLG(txt,'information',['CUSUM - ' PlotType],500,150,CurrINI,1);
WaitControls = get(WaitHandle,'UserData');
%
% compute the mean of X for the full sample
%
mu = (1/(NumIter-NumBurnin))*sum(X,1);
%
% create the difference from the mean over all values for the statistics
%
Xbar = X-(ones(NumIter-NumBurnin,1)*mu);
%
% accumulate and add the full sample mean
%
CUSUMPost = cumsum(Xbar,1)+(ones(NumIter-NumBurnin,1)*mu);
drawnow;
%
StopEstimation = 0;
NumSum = ceil(0.1*(NumIter-NumBurnin));
if NumIter-NumBurnin-NumSum>0;
   CUSUMAverage = zeros(NumIter-NumBurnin-NumSum,NumStat);
   for i=2:NumIter-NumBurnin-NumSum+1;
      %
      % check if the cancel button has been pressed
      %
      if strcmp(get(WaitControls.cancel,'UserData'),'cancel')==1;
         StopEstimation = 1;
         CUSUMAverage = [];
         break;
      end;
      CUSUMAverage(i,:) = sum(Xbar(i:NumSum+i-1,:),1)+mu;
      drawnow;
   end;
end;
%
% shut down the wait dialog
%
set(WaitControls.text,'UserData','done');
delete(WaitHandle);
drawnow;
pause(0.02);
%
% check if we should set CUSUMPost to an empty matrix
%
if StopEstimation==1;
   txt = ['The CUSUM estimates for all posterior draws of the ' lower(PlotType) ' have already been calculated, while the CUSUM estimates for the moving window of length ' num2str(NumSum,'%0.0f') ' was aborted. Would you still like to view the CUSUM estimates based on all posterior draws?'];
   answer = Query(txt,'question',200,['CUSUM - ' PlotType],500,'no',CurrINI);
   drawnow;
   if strcmp(lower(answer),'no')==1;
      CUSUMPost = [];
   end;
end;

%
% end of CUSUM.m
%
