function ResStr = StrWrap(InStr,StrLen)
% StrWrap: Creates a string matrix with a user specified maximum length
%
% USAGE:
%
%       ResStr = StrWrap(InStr,StrLen)
%
% REQUIRED INPUT:  InStr (string vector) that should be warpped if length is greater than StrLen.
%
%                  StrLen (integer), determining the maximum length of the output string matrix.
%
% REQUIRED OUTPUT: ResStr (string matrix), with the possibly wrapped InStr string vector.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: August 24, 2006.
%                        This version: May 23, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 21-11-2006: Fixed a bug which occurred when, for instance, the length of the string was greater than
%               the maximum allowed, but the last position of a space character was lower than the length
%               of the string. This issue is handled through the use of the vector "StrPrtsLen" and the
%               "ExtraTerm" variable that uses it.
%
% * 07-11-2007: Updated the documentation.
%
% * 07-12-2007: Added the length of the last entry to "StrPrtsLen". This fixed a bug that occured if the space position prior
%               to the last entry was less than StrLen, while the total number of chcracters was greater than StrLen, then
%               the wrap position would not be found.
%
% * 12-12-2007: If WrapPositions was empty with the length of InStr being greater than Strlen, then
%               the WrapPositions variable is set to the last space character.
%
% * 23-05-2008: Updated the documentation.
%
% * 22-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% If InStr already satisfies StrLen max value, return InStr as output
%
if size(InStr,2)<=StrLen;
   ResStr = InStr;
   return;
end;
%
% check if there as space characters
%
InStr = StringTrim(InStr(1,:));
NumSpaces = findstr(InStr,' ');
if isempty(NumSpaces);
   %
   % there are no natural places to wrap the string
   %
   ResStr = InStr;
   return;
end;
%
NumChars = size(InStr,2);
WrapPositions = [];
LastWrap = StrLen;
ResStr = '';
if length(NumSpaces)>0;
   %
   % count the maximum length of each entry
   %
   StrPrtsLen = [NumSpaces(1,1)-1 NumSpaces(1,2:length(NumSpaces))-NumSpaces(1,1:length(NumSpaces)-1)-1 length(InStr)-NumSpaces(1,length(NumSpaces))];
end;
for i=1:length(NumSpaces);
   %
   % search for break-points
   %
   if NumSpaces(i)>LastWrap;
      if i>1;
         WrapPositions = [WrapPositions NumSpaces(i-1)];
         LastWrap = StrLen+NumSpaces(i-1);
      else;
         WrapPositions = NumSpaces(i);
         LastWrap = StrLen+NumSpaces(i);
      end;
   elseif NumSpaces(i)+StrPrtsLen(i+1)>=LastWrap;
      WrapPositions = [WrapPositions NumSpaces(i)];
      LastWrap = StrLen+NumSpaces(i);
   end;
end;
if isempty(WrapPositions)==1;
   WrapPositions = NumSpaces(length(NumSpaces));
end;
for i=1:length(WrapPositions);
   if i>1;
      ResStr = strvcat(ResStr,StringTrim(InStr(1,WrapPositions(i-1):WrapPositions(i))));
   else;
      %
      % initialize the ResStr
      %
      ResStr = StringTrim(InStr(1,1:WrapPositions(1)));
   end;
end;
%
% add the last line
%
ResStr = strvcat(ResStr,StringTrim(InStr(1,WrapPositions(length(WrapPositions)):NumChars)));

%
% end of StrWrap.m
%
