function PrintStateVarDistribution(ResFile,SVEs,DSGEModel,CurrINI,Selector,NumDraws,TotalDraws,CurrChain,NumPaths)
% PrintStateVarDistribution: Prints results for state variable correlations, state shock correlations,
%                            and measurement error correlations to file.
%
% USAGE:
%
%       PrintStateVarDistribution(ResFile,SVEs,DSGEModel,CurrINI,Selector,NumDraws,TotalDraws,CurrChain,NumPaths)
%
% REQUIRED INPUT: ResFile (string vector) with the full path+name of the file to be printed.
%
%                 SVEs (structure) with fields that depend on the "Selector" input. If the results
%                      concern state variable correlations, then the fields are "StdKsi" and "CorrKsi".
%                      If the results concern state shock correlations, then the fields are "Meaneta",
%                      "Stdeta", "Correta", and "StateShocks". If instead the results are for measurement error
%                      correlations, then the fields are "Meanw", "Stdw", "Corrw", and "ErrorIndicator".
%                      The field "Quantiles" are present for all, with its "Selector" depends subfields
%                      and the common field "percent".
%
%                 DSGEModel (structure) with fields containing all the information about a DSGE Model for YADA.
%
%                 CurrINI (structure) with initialization information.
%
%                 Selector (integer) that is:
%                                  1 - if the function should work with state variable correlations;
%                                  2 - if the function should work with state shock correlations; and
%                                  3 - if the function should work with measurement error correlations.
%                                  4 - if the function should work with the data and model based means
%                                      and standard deviations.
%
%                  NumDraws (integer) with the number of effective draws from the posterior distribution of the
%                                     parameters.
%
%                  TotalDraws (integer) with the number of draws from the posterior distribution of the parameters
%                                     that could have been used.
%
%                  CurrChain (integer) with the number for the current Markov chain.
%
%                  NumPaths (integer) with the number of data paths to simulate per parameter value.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: November 27, 2007.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 28-11-2007: Added the code for the measurement errors and the state variables.
%
% * 30-11-2007: Added the code for the data and model based means and standard deviations.
%
% * 28-2-2008:  Added the variable "NumPaths" to allow printing results from simulated data.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% open the file
%
fid = fopen(ResFile,'wt');
%
% initializing the formatting of output numbers. we usually apply the 12.6f format.
%
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
%
% select what to print
%
if Selector==1;
   fprintf(fid,'********************************************************************************\n');
   fprintf(fid,'*                                                                              *\n');
   if NumPaths==0;
      fprintf(fid,'*                       P O P U L A T I O N - B A S E D                        *\n');
      fprintf(fid,'*                        S T A T E   V A R I A B L E S                         *\n');
   else;
      fprintf(fid,'*                       S I M U L A T I O N - B A S E D                        *\n');
      fprintf(fid,'*                  S T A T E   V A R I A B L E   M E A N S ,                   *\n');
   end;
   fprintf(fid,'*   S T A N D A R D   D E V i A T I O N S   A N D   C O R R E L A T I O N S    *\n');
   fprintf(fid,'*                                                                              *\n');
   fprintf(fid,'********************************************************************************\n\n');
   %
   fprintf(fid,'Number of parameter draws used:      %0.0f\n',NumDraws);
   fprintf(fid,'Number of available parameter draws: %0.0f\n',TotalDraws);
   if NumPaths>0;
      fprintf(fid,'Number of paths per parameter draw:  %0.0f\n',NumPaths);
   end;
   fprintf(fid,'Current MCMC chain:                  %0.0f\n\n',CurrChain);
   [r,vn] = size(DSGEModel.StateVariableNames);
   if vn<20;
      AddNameStr = SpaceStr(20-vn);
   else;
      AddNameStr = '';
   end;
   if NumPaths==0;
      prt_mean_str = ['%s%s           ' prt_val '\n'];
      prt_quant_str = ['%3.0f percent            [' prt_val ',' prt_val '] \n'];
      %
      % determine all the quantile data
      %
      NumBounds = floor(length(SVEs.Quantiles)/2);
      QuantVec = zeros(NumBounds,1);
      QuantStd = zeros(NumBounds,2,r);
      for nb=1:NumBounds;
         QuantVec(NumBounds-nb+1,1) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-SVEs.Quantiles(nb).percent;
         for i=1:r;
            QuantStd(NumBounds-nb+1,1,i) = SVEs.Quantiles(nb).StdKsi(i,1);
            QuantStd(NumBounds-nb+1,2,i) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).StdKsi(i,1);
         end;
      end;
      %
      % header for the mean and the standard deviation
      %
      fprintf(fid,'===================================================\n');
      fprintf(fid,'      State variable       Standard deviation\n');
      fprintf(fid,'===================================================\n');
      for i=1:r;
         fprintf(fid,prt_mean_str,AddNameStr,DSGEModel.StateVariableNames(i,:),SVEs.StdKsi(i,1));
         for j=1:NumBounds;
            fprintf(fid,prt_quant_str,QuantVec(j,1),QuantStd(j,:,i)');
         end;
         if i<r;
            fprintf(fid,'---------------------------------------------------\n');
         else;
            fprintf(fid,'===================================================\n\n');
         end;
      end;
   else;
      prt_mean_str = ['%s%s           ' prt_val '                  ' prt_val '\n'];
      prt_quant_str = ['%3.0f percent            [' prt_val ',' prt_val ']   [' prt_val ',' prt_val '] \n'];
      %
      % determine all the quantile data
      %
      NumBounds = floor(length(SVEs.Quantiles)/2);
      QuantVec = zeros(NumBounds,1);
      QuantStd = zeros(NumBounds,4,r);
      for nb=1:NumBounds;
         QuantVec(NumBounds-nb+1,1) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-SVEs.Quantiles(nb).percent;
         for i=1:r;
            QuantStd(NumBounds-nb+1,1,i) = SVEs.Quantiles(nb).MeanKsi(i,1);
            QuantStd(NumBounds-nb+1,2,i) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).MeanKsi(i,1);
            QuantStd(NumBounds-nb+1,3,i) = SVEs.Quantiles(nb).StdKsi(i,1);
            QuantStd(NumBounds-nb+1,4,i) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).StdKsi(i,1);
         end;
      end;
      %
      % header for the mean and the standard deviation
      %
      fprintf(fid,'=================================================================================\n');
      fprintf(fid,'      State variable               Mean                  Standard deviation\n');
      fprintf(fid,'=================================================================================\n');
      for i=1:r;
         fprintf(fid,prt_mean_str,AddNameStr,DSGEModel.StateVariableNames(i,:),SVEs.MeanKsi(i,1),SVEs.StdKsi(i,1));
         for j=1:NumBounds;
            fprintf(fid,prt_quant_str,QuantVec(j,1),QuantStd(j,:,i)');
         end;
         if i<r;
            fprintf(fid,'---------------------------------------------------------------------------------\n');
         else;
            fprintf(fid,'=================================================================================\n\n');
         end;
      end;
   end;
   %
   % and now the correlations
   %
   if r>1;
      ShockStrMat = '';
      CorrQuant = zeros(NumBounds,2,r*(r-1)/2);
      CorrKsi = zeros(r*(r-1)/2,1);
      num = 1;
      for j=1:r-1;
         for i=j+1:r;
            ShockStrMat = strvcat(ShockStrMat,[DSGEModel.StateVariableNames(j,:) ' ' DSGEModel.StateVariableNames(i,:)]);
            for nb=1:NumBounds;
               CorrQuant(NumBounds-nb+1,1,num) = SVEs.Quantiles(nb).CorrKsi(j,i);
               CorrQuant(NumBounds-nb+1,2,num) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).CorrKsi(j,i);
            end;
            CorrKsi(num,1) = SVEs.CorrKsi(j,i);
            num = num+1;
         end;
      end;
      prt_corr_str = ['%s%s%s           ' prt_val ' \n'];
      prt_quant_str = ['                 %3.0f percent                [' prt_val ',' prt_val '] \n'];
      %
      fprintf(fid,'========================================================================\n');
      fprintf(fid,'          Variable 1/Variable 2                      Correlation\n');
      fprintf(fid,'========================================================================\n');
      for i=1:r*(r-1)/2;
         fprintf(fid,prt_corr_str,AddNameStr,ShockStrMat(i,:),AddNameStr,CorrKsi(i));
         for j=1:NumBounds;
            fprintf(fid,prt_quant_str,QuantVec(j,1),CorrQuant(j,:,i)');
         end;
         if i<r*(r-1)/2;
            fprintf(fid,'------------------------------------------------------------------------\n');
         else;
            fprintf(fid,'========================================================================\n\n');
         end;
      end;
   end;
elseif Selector==2;
   fprintf(fid,'********************************************************************************\n');
   fprintf(fid,'*                                                                              *\n');
   fprintf(fid,'*                S M O O T H   E S T I M A T I O N - B A S E D                 *\n');
   fprintf(fid,'*      S T A T E   S H O C K   M E A N   A N D   C O R R E L A T I O N S       *\n');
   fprintf(fid,'*                                                                              *\n');
   fprintf(fid,'********************************************************************************\n\n');
   %
   fprintf(fid,'Number of parameter draws used:      %0.0f\n',NumDraws);
   fprintf(fid,'Number of available parameter draws: %0.0f\n',TotalDraws);
   fprintf(fid,'Current MCMC chain:                  %0.0f\n\n',CurrChain);
   %
   [n,vn] = size(SVEs.ShockNames);
   if vn<15;
      AddNameStr = SpaceStr(15-vn);
   else;
      AddNameStr = '';
   end;
   q = length(SVEs.Meaneta);
   prt_mean_str = ['%s%s           ' prt_val '                  ' prt_val ' \n'];
   prt_quant_str = ['%3.0f percent       [' prt_val ',' prt_val ']   [' prt_val ',' prt_val '] \n'];
   %
   % determine all the quantile data
   %
   NumBounds = floor(length(SVEs.Quantiles)/2);
   QuantVec = zeros(NumBounds,1);
   QuantMean = zeros(NumBounds,2,q);
   QuantStd = zeros(NumBounds,2,q);
   for nb=1:NumBounds;
      QuantVec(NumBounds-nb+1,1) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-SVEs.Quantiles(nb).percent;
      for i=1:q;
         QuantMean(NumBounds-nb+1,1,i) = SVEs.Quantiles(nb).Meaneta(i,1);
         QuantMean(NumBounds-nb+1,2,i) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Meaneta(i,1);
         QuantStd(NumBounds-nb+1,1,i) = SVEs.Quantiles(nb).Stdeta(i,1);
         QuantStd(NumBounds-nb+1,2,i) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Stdeta(i,1);
      end;
   end;
   %
   % header for the mean and the standard deviation
   %
   fprintf(fid,'============================================================================\n');
   fprintf(fid,'          Shock               Mean                  Standard deviation\n');
   fprintf(fid,'============================================================================\n');
   for i=1:q;
      fprintf(fid,prt_mean_str,AddNameStr,SVEs.ShockNames(i,:),SVEs.Meaneta(i,1),SVEs.Stdeta(i,1));
      for j=1:NumBounds;
         fprintf(fid,prt_quant_str,QuantVec(j,1),QuantMean(j,:,i)',QuantStd(j,:,i)');
      end;
      if i<q;
         fprintf(fid,'----------------------------------------------------------------------------\n');
      else;
         fprintf(fid,'============================================================================\n\n');
      end;
   end;
   %
   % and now the correlations
   %
   if q>1;
      ShockStrMat = '';
      CorrQuant = zeros(NumBounds,2,q*(q-1)/2);
      Correta = zeros(q*(q-1)/2,1);
      num = 1;
      for j=1:q-1;
         for i=j+1:q;
            ShockStrMat = strvcat(ShockStrMat,[SVEs.ShockNames(j,:) ' ' SVEs.ShockNames(i,:)]);
            for nb=1:NumBounds;
               CorrQuant(NumBounds-nb+1,1,num) = SVEs.Quantiles(nb).Correta(j,i);
               CorrQuant(NumBounds-nb+1,2,num) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Correta(j,i);
            end;
            Correta(num,1) = SVEs.Correta(j,i);
            num = num+1;
         end;
      end;
      prt_corr_str = ['%s%s%s           ' prt_val ' \n'];
      prt_quant_str = ['            %3.0f percent           [' prt_val ',' prt_val '] \n'];
      %
      fprintf(fid,'==============================================================\n');
      fprintf(fid,'        Shock 1/Shock 2                    Correlation\n');
      fprintf(fid,'==============================================================\n');
      for i=1:q*(q-1)/2;
         fprintf(fid,prt_corr_str,AddNameStr,ShockStrMat(i,:),AddNameStr,Correta(i));
         for j=1:NumBounds;
            fprintf(fid,prt_quant_str,QuantVec(j,1),CorrQuant(j,:,i)');
         end;
         if i<q*(q-1)/2;
            fprintf(fid,'--------------------------------------------------------------\n');
         else;
            fprintf(fid,'==============================================================\n\n');
         end;
      end;
   end;
elseif Selector==3;
   %
   % get the variables with non-zero measurement errors
   %
   SelectVariables = SVEs.ErrorIndicator .* (1:length(SVEs.ErrorIndicator))';
   SelectVariables = SelectVariables(SelectVariables>0);
   fprintf(fid,'********************************************************************************\n');
   fprintf(fid,'*                                                                              *\n');
   fprintf(fid,'*                S M O O T H   E S T I M A T I O N - B A S E D                 *\n');
   fprintf(fid,'*                      M E A S U R E M E N T   E R R O R                       *\n');
   fprintf(fid,'*                  M E A N   A N D   C O R R E L A T I O N S                   *\n');
   fprintf(fid,'*                                                                              *\n');
   fprintf(fid,'********************************************************************************\n\n');
   %
   fprintf(fid,'Number of parameter draws used:      %0.0f\n',NumDraws);
   fprintf(fid,'Number of available parameter draws: %0.0f\n',TotalDraws);
   fprintf(fid,'Current MCMC chain:                  %0.0f\n\n',CurrChain);
   [n,vn] = size(DSGEModel.VariableNames);
   if vn<20;
      AddNameStr = SpaceStr(20-vn);
   else;
      AddNameStr = '';
   end;
   VariableNames = DSGEModel.VariableNames(SelectVariables,:);
   %
   n = length(SVEs.Meanw);
   %
   prt_mean_str = ['%s%s           ' prt_val '                  ' prt_val ' \n'];
   prt_quant_str = ['%3.0f percent            [' prt_val ',' prt_val ']   [' prt_val ',' prt_val '] \n'];
   %
   % determine all the quantile data
   %
   NumBounds = floor(length(SVEs.Quantiles)/2);
   QuantVec = zeros(NumBounds,1);
   QuantMean = zeros(NumBounds,2,n);
   QuantStd = zeros(NumBounds,2,n);
   for nb=1:NumBounds;
      QuantVec(NumBounds-nb+1,1) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-SVEs.Quantiles(nb).percent;
      for i=1:n;
         QuantMean(NumBounds-nb+1,1,i) = SVEs.Quantiles(nb).Meanw(i,1);
         QuantMean(NumBounds-nb+1,2,i) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Meanw(i,1);
         QuantStd(NumBounds-nb+1,1,i) = SVEs.Quantiles(nb).Stdw(i,1);
         QuantStd(NumBounds-nb+1,2,i) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Stdw(i,1);
      end;
   end;
   %
   % header for the mean and the standard deviation
   %
   fprintf(fid,'=================================================================================\n');
   fprintf(fid,'        Error in:                    Mean                  Standard deviation\n');
   fprintf(fid,'=================================================================================\n');
   for i=1:n;
      fprintf(fid,prt_mean_str,AddNameStr,VariableNames(i,:),SVEs.Meanw(i,1),SVEs.Stdw(i,1));
      for j=1:NumBounds;
         fprintf(fid,prt_quant_str,QuantVec(j,1),QuantMean(j,:,i)',QuantStd(j,:,i)');
      end;
      if i<n;
         fprintf(fid,'---------------------------------------------------------------------------------\n');
      else;
         fprintf(fid,'=================================================================================\n\n');
      end;
   end;
   %
   % and now the correlations
   %
   if n>1;
      ShockStrMat = '';
      CorrQuant = zeros(NumBounds,2,n*(n-1)/2);
      Corrw = zeros(n*(n-1)/2,1);
      num = 1;
      for j=1:n-1;
         for i=j+1:n;
            ShockStrMat = strvcat(ShockStrMat,[VariableNames(j,:) ' ' VariableNames(i,:)]);
            for nb=1:NumBounds;
               CorrQuant(NumBounds-nb+1,1,num) = SVEs.Quantiles(nb).Corrw(j,i);
               CorrQuant(NumBounds-nb+1,2,num) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Corrw(j,i);
            end;
            Corrw(num,1) = SVEs.Corrw(j,i);
            num = num+1;
         end;
      end;
      prt_corr_str = ['%s%s%s           ' prt_val ' \n'];
      prt_quant_str = ['                 %3.0f percent                [' prt_val ',' prt_val '] \n'];
      %
      fprintf(fid,'========================================================================\n');
      fprintf(fid,'             Error 1/Error 2                         Correlation\n');
      fprintf(fid,'========================================================================\n');
      for i=1:n*(n-1)/2;
         fprintf(fid,prt_corr_str,AddNameStr,ShockStrMat(i,:),AddNameStr,Corrw(i));
         for j=1:NumBounds;
            fprintf(fid,prt_quant_str,QuantVec(j,1),CorrQuant(j,:,i)');
         end;
         if i<n*(n-1)/2;
            fprintf(fid,'------------------------------------------------------------------------\n');
         else;
            fprintf(fid,'========================================================================\n\n');
         end;
      end;
   end;
elseif Selector==4;
   fprintf(fid,'********************************************************************************\n');
   fprintf(fid,'*                                                                              *\n');
   if NumPaths==0;
      fprintf(fid,'*                       P O P U L A T I O N - B A S E D                        *\n');
   else;
      fprintf(fid,'*                           S A M P L E - B A S E D                            *\n');
   end;
   fprintf(fid,'*                 M E A N S   A N D   C O R R E L A T I O N S                  *\n');
   fprintf(fid,'*                                                                              *\n');
   fprintf(fid,'********************************************************************************\n\n');
   %
   fprintf(fid,'Number of parameter draws used:      %0.0f\n',NumDraws);
   fprintf(fid,'Number of available parameter draws: %0.0f\n',TotalDraws);
   if NumPaths>0;
      fprintf(fid,'Number of paths per parameter draw:  %0.0f\n',NumPaths);
   end;
   fprintf(fid,'Current MCMC chain:                  %0.0f\n\n',CurrChain);
   %
   [n,vn] = size(DSGEModel.VariableNames);
   if vn<20;
      AddNameStr = SpaceStr(20-vn);
   else;
      AddNameStr = '';
   end;
   %
   % deal with the means
   %
   NumBounds = floor(length(SVEs.Quantiles)/2);
   QuantVec = zeros(NumBounds,1);
   QuantStd = zeros(NumBounds,2,n);
   if NumPaths==0;
      if min(size(DSGEModel.X))>0;
         k = size(DSGEModel.X,1);
         for j=1:k;
            Quant(j).Mean = zeros(NumBounds,2,n);
         end;
      else;
         k = 0;
      end;
   else;
      k = size(SVEs.Mean,2);
      for j=1:k;
         Quant(j).Mean = zeros(NumBounds,2,n);
      end;
   end;
   for nb=1:NumBounds;
      QuantVec(NumBounds-nb+1,1) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-SVEs.Quantiles(nb).percent;
      for i=1:n;
         QuantStd(NumBounds-nb+1,1,i) = SVEs.Quantiles(nb).Std(i,1);
         QuantStd(NumBounds-nb+1,2,i) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Std(i,1);
         for j=1:k;
            Quant(j).Mean(NumBounds-nb+1,1,i) = SVEs.Quantiles(nb).Mean(i,j);
            Quant(j).Mean(NumBounds-nb+1,2,i) = SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Mean(i,j);
         end;
      end;
   end;
   %
   % define some print string
   %
   prt_std_str = ['%s%s           ' prt_val '          |   ' prt_val ' \n'];
   prt_quant_str = ['%3.0f percent            [' prt_val ',' prt_val ']   |\n'];
   if k>0;
      kn = size(DSGEModel.XVariableNames,2);
      if kn<20;
         AddXNameStr = SpaceStr(20-kn);
      else;
         AddXNameStr = '';
      end;
      %
      % print the mean results
      %
      for j=1:k;
         fprintf(fid,'======================================================================\n');
         fprintf(fid,'                                                  Averages \n');
         fprintf(fid,'                                        Variable: %s%s \n',DSGEModel.XVariableNames(j,:),AddXNameStr);
         fprintf(fid,'Variable                         DSGE Model          |       Data \n');
         fprintf(fid,'======================================================================\n');
         for i=1:n;
            fprintf(fid,prt_std_str,AddNameStr,DSGEModel.VariableNames(i,:),SVEs.Mean(i,j),SVEs.Data.Mean(i,j));
            for nb=1:NumBounds;
               fprintf(fid,prt_quant_str,QuantVec(nb,1),Quant(j).Mean(nb,:,i)');
            end;
            if i<n;
               fprintf(fid,'----------------------------------------------------------------------\n');
            else;
               fprintf(fid,'======================================================================\n\n');
            end;
         end;
      end;
   end;
   %
   % the standard deviations
   %
   fprintf(fid,'======================================================================\n');
   fprintf(fid,'                                             Standard Deviations \n');
   fprintf(fid,'Variable                         DSGE Model          |       Data \n');
   fprintf(fid,'======================================================================\n');
   for i=1:n;
      fprintf(fid,prt_std_str,AddNameStr,DSGEModel.VariableNames(i,:),SVEs.Std(i,1),SVEs.Data.Std(i,1));
      for j=1:NumBounds;
         fprintf(fid,prt_quant_str,QuantVec(j,1),QuantStd(j,:,i)');
      end;
      if i<n;
         fprintf(fid,'----------------------------------------------------------------------\n');
      else;
         fprintf(fid,'======================================================================\n\n');
      end;
   end;
end;
%
% bottom file data
%
fprintf(fid,'Directory for file: %s\n',GetPath(ResFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(ResFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintStateVarDistribution.m
%
