function PrintStateCorrelations(DSGEModel,CorrData,CurrINI,EstStr)
% PrintStateCorrelations: Writes the results from calculating the variances and contemporaneous correlations
%                         of the state variables to disk.
%
% USAGE:
%
%       PrintStateCorrelations(DSGEModel,CorrData,CurrINI,EstStr)
%
% REQUIRED INPUT: DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 CorrData (structure) with main fields "StateSpace" and "Data". These fields have sub-fields
%                           "Mean", "AutoCov", and "AutoCorr". The main field "StateSpace" refers to the
%                           results based on the state-space model, while the main field "Data" refers to
%                           the results based on the data. Notice that the autocovariance and autocorrelation
%                           results using the data are calculated by de-meaning using the mean from the
%                           state-space model. This should make the results across the two approaches
%                           easier to compare.
%
%                 CurrINI (structure), with initialization information.
%
%                 EstStr (string vector) that takes the value "posterior mode" or "initial".
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: October 31, 2006.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(lower(EstStr),'posterior mode')==1;
   ResFile = [DSGEModel.OutputDirectory '\PosteriorModeStateCorr-' DSGEModel.NameOfModel '.txt'];
else;
   ResFile = [DSGEModel.OutputDirectory '\InitialValueStateCorr-' DSGEModel.NameOfModel '.txt'];
end;
fid = fopen(ResFile,'wt');
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*            S T A T E   V A R I A B L E   C O R R E L A T I O N S             *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
fprintf(fid,'NOTE: Using the %s values.\n\n',lower(EstStr));
%
% initializing the formatting of output numbers. we usually apply the 12.6f format.
%
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
%
[r,vn] = size(DSGEModel.StateVariableNames);
if vn<20;
   add_name_str = SpaceStr(20-vn);
else;
   add_name_str = '';
end;
fprintf(fid,'============================================\n');
fprintf(fid,'            Variable   Standard Deviation\n');
fprintf(fid,'============================================\n');
for i=1:r;
   fprintf(fid,['%s%s   ' prt_val '\n'],add_name_str,DSGEModel.StateVariableNames(i,:),CorrData.StateVars.StDev(i));
   if i<r;
      fprintf(fid,'--------------------------------------------\n');
   end;
end;
fprintf(fid,'============================================\n\n');
%
% and now for the correlations
%
if r>1;
   fprintf(fid,'===========================================================\n');
   fprintf(fid,'            Variable             Variable    Correlation\n');
   fprintf(fid,'===========================================================\n');
end;
for i=1:r;
   for j=i+1:r;
      fprintf(fid,['%s%s %s%s   ' prt_val '\n'],add_name_str,DSGEModel.StateVariableNames(i,:),add_name_str,DSGEModel.StateVariableNames(j,:),CorrData.StateVars.Correlations(i,j));
      if (i*j)~=((r-1)*r);
         fprintf(fid,'-----------------------------------------------------------\n');
      else;
         fprintf(fid,'===========================================================\n\n');
      end;
   end;
end;
%
% bottom file data
%
fprintf(fid,'Directory for file: %s\n',GetPath(ResFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(ResFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintStateCorrelations.m
%
