function PrintSeparatedPartialMeans(GewekeStat,p,CurrChain,NumChains,NumIter,NumBurnin,PlotType,NamesMatrix,DSGEModel,CurrINI)
% PrintSeparatedPartialMeans: Prints the separated partial means tests of Geweke to text file.
%
% USAGE:
%
%       PrintSeparatedPartialMeans(GewekeStat,p,CurrChain,NumChains,NumIter,NumBurnin,PlotType,NamesMatrix,DSGEModel,CurrINI)
%
% REQUIRED INPUT: GewekeStat (matrix) of dimension NumParam x 2 where the first column
%                            holds the values of the NumParam different separated partial
%                            means tests, while the second column holds the p-values from
%                            the chi-square distribution with p-1 degrees of freedom.
%
%                 p (integer) with the number of partial means.
%
%                 CurrChain (integer) with the number for the current MCMC chain.
%
%                 NumChains (integer) with the total number of MCMC chains available.
%
%                 NumIter (integer) with the total number of posterior draws.
%
%                 NumBurnin (integer) with the number of burn-in draws.
%
%                 PlotType (string vector) that takes on the values "log posterior",
%                            "original parameters", and "transformed parameters".
%
%                 NamesMatrix (string matrix) with the names for the NumParam parameters that
%                            have been tested.
%
%                 DSGEModel (structure)  whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: April 19, 2007.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 20-4-2007:  Finished the function.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% first we open a text file
%
ResFile = [DSGEModel.OutputDirectory '\SeparatedPartialMeans-' strrep(PlotType,' ','') '-' DSGEModel.NameOfModel '-' num2str(p,'%0.0f') '-' num2str(CurrChain,'%0.0f') '.' num2str(NumChains,'%0.0f') '-' num2str(NumBurnin,'%0.0f') '.' num2str(NumIter,'%0.0f') '.txt'];
fid = fopen(ResFile,'wt');
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*           S E P A R A T E D   P A R T I A L   M E A N S   T E S T            *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% some data to consider first
%
NumParam = size(GewekeStat,1);
Np = (NumIter-NumBurnin)/(2*p);
fprintf(fid,'Number of posterior draws:             %0.0f\n',NumIter);
fprintf(fid,'Number of burn-in draws:               %0.0f\n',NumBurnin);
fprintf(fid,'MCMC chain number:                     %0.0f\n',CurrChain);
if strcmp(lower(PlotType),'log posterior')==0;
   fprintf(fid,'Number of parameters to estimate:      %0.0f\n',NumParam);
end;
fprintf(fid,'Number of draws per partial mean:      %0.0f\n',Np);
fprintf(fid,'Number of separated partial means:     %0.0f\n\n',p);
%
% we can now print the results
%
if strcmp(lower(PlotType),'log posterior')==1;
   %
   % we have the log posterior to considet
   %
   fprintf(fid,'Geweke''s separated partial means test: %0.6f\n',GewekeStat(1,1));
   fprintf(fid,'with asymptotic p-value:               %0.6f\n\n',GewekeStat(1,2));
   fprintf(fid,'================================================================================\n\n');
else;
   %
   % we're dealing with parameters
   %
   if size(NamesMatrix,2)<20;
      SpStr = SpaceStr(20-size(NamesMatrix,2));
   else;
      SpStr = '';
   end;
   prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
   prt_str = ['%s%s   ' prt_val '   ' prt_val];
   fprintf(fid,'===================================================\n');
   fprintf(fid,'            Geweke''s separated partial means test\n');
   fprintf(fid,'           parameter       Test           p-value \n');
   for i=1:NumParam;
      if i==1;
         fprintf(fid,'===================================================\n');
      else;
         fprintf(fid,'---------------------------------------------------\n');
      end;
      fprintf(fid,[prt_str '\n'],SpStr,NamesMatrix(i,:),GewekeStat(i,1),GewekeStat(i,2));
   end;
   fprintf(fid,'===================================================\n\n');
end;
%
% bottom file data
%
fprintf(fid,'Directory for file: %s\n',GetPath(ResFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(ResFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintSeparatedPartialMeans.m
%
