function TextFile = PrintPriorInformation(ParameterNames,theta,thetaDist,PriorDist,DSGEModel,CurrINI)
% PrintPriorInformation: Prints information about parameters to be estimated using the data in the prior file.
%
% USAGE:
%
%       PrintPriorInformation(ParameterNames,theta,thetaDist,PriorDist,DSGEModel,CurrINI)
%
% REQUIRED INPUT:  ParameterNames (structure) with fields "calibrated", "beta", "gamma", "normal", "invgamma", "truncnormal",
%                                 "uniform", "all" and "estimated". Each field returns a string matrix with the parameter names.
%
%                  theta (vector) with initial values for the parameters to be estimated.
%
%                  thetaDist (vector) with integers 1 if gamma, 2 if beta, 3 if normal, 4 if invgamma, 5 if
%                                 truncnormal, and 6 if uniform.
%
%                  PriorDist (structure) with fields "beta", "beta_ab", "gamma", "gamma_ab", "normal", "invgamma",
%                                 "truncnormal", and "uniform" where the values are matrices with the parameter values for the
%                                 distributions. In the case of the normal, beta, and gamma these parameter values are
%                                 the mean and the standard deviation. For the invgamma it is the s and q parameters,
%                                 where q is a positive integer (degrees of freedom). For the left truncated normal
%                                 there's a third parameter, the lower bound. Finally, for the uniform the parameters
%                                 are the lower and the upper bound. For the beta, gamma and inverse gamma there is a
%                                 third column with the lower bound values, while for the beta there is also a fourth
%                                 column with the upper bound.
%
%                  DSGEModel (structure) whose fields contain all required information about the model, such as input
%                                 files, variable and sample data. Actual observations are not stored in this structure.
%
%                  CurrINI (structure) with initialization information.
%
% REQUIRED OUTPUT: TextFile (string vector) with the name of the output file.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: July 24, 2008.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if isempty(DSGEModel.NameOfModel)==0;
   TextFile = [DSGEModel.OutputDirectory '\PriorInformation-' DSGEModel.NameOfModel '.txt'];
else;
   TextFile = [DSGEModel.OutputDirectory '\PriorInformation.txt'];
end;
%
fid = fopen(TextFile,'wt');
%
% the header information
%
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*         P R I O R   D I S T R I B U T I O N   I N F O R M A T I O N          *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
fprintf(fid,'Prior distribution data file:     %s\n',GetFilename(DSGEModel.PriorFile));
fprintf(fid,'Sheet name:                       %s\n\n',DSGEModel.PriorFileSheet);
%
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
%
% check the length of parameter names string
%
ns = size(ParameterNames.estimated,2);
if ns<20;
   add_str = SpaceStr(20-ns);
else;
   add_str = '';
end;
%
if (isempty(PriorDist.beta)==0)|(isempty(PriorDist.uniform)==0);
   fprintf(fid,'====================================================================================================================\n');
   fprintf(fid,'                                                Original Parameters                                                 \n');
   fprintf(fid,'      parameter          prior dist   initial value     parameter 1     parameter 2     lower bound     upper bound\n');
   SepStr = '--------------------------------------------------------------------------------------------------------------------';
   EndStr = '====================================================================================================================';
else;
   if (isempty(PriorDist.gamma)==0)|(isempty(PriorDist.invgamma)==0)|(isempty(PriorDist.truncnormal)==0);
      fprintf(fid,'====================================================================================================\n');
      fprintf(fid,'                                        Original Parameters                                         \n');
      fprintf(fid,'      parameter          prior dist   initial value     parameter 1     parameter 2     lower bound\n');
      SepStr = '----------------------------------------------------------------------------------------------------';
      EndStr = '====================================================================================================';
   else;
      fprintf(fid,'====================================================================================\n');
      fprintf(fid,'                                        Original Parameters                         \n');
      fprintf(fid,'      parameter          prior dist   initial value     parameter 1     parameter 2\n');
      SepStr = '------------------------------------------------------------------------------------';
      EndStr = '====================================================================================';
   end;
end;
%
% initialize parameters
%
n_gamma = 0;
n_beta = 0;
n_normal = 0;
n_invgamma = 0;
n_truncnormal = 0;
n_uniform = 0;
%
% initialize print string
%
PrtStr = ['%s%s    %s%s    ' prt_val '    ' prt_val '    ' prt_val];
for i=1:length(theta);
   if thetaDist(i)==1;
      dist_str = 'gamma';
      dist_add = SpaceStr(6);
      n_gamma = n_gamma+1;
      prior_val = PriorDist.gamma(n_gamma,:);
   elseif thetaDist(i)==2;
      dist_str = 'beta';
      dist_add = SpaceStr(7);
      n_beta = n_beta+1;
      prior_val = PriorDist.beta(n_beta,:);
   elseif thetaDist(i)==3;
      dist_str = 'normal';
      dist_add = SpaceStr(5);
      n_normal = n_normal+1;
      prior_val = PriorDist.normal(n_normal,:);
   elseif thetaDist(i)==4;
      dist_str = 'invgamma';
      dist_add = SpaceStr(3);
      n_invgamma = n_invgamma+1;
      prior_val = PriorDist.invgamma(n_invgamma,:);
   elseif thetaDist(i)==5;
      dist_str = 'truncnormal';
      dist_add = '';
      n_truncnormal = n_truncnormal+1;
      prior_val = PriorDist.truncnormal(n_truncnormal,:);
   elseif thetaDist(i)==6;
      dist_str = 'uniform';
      dist_add = SpaceStr(4);
      n_uniform = n_uniform+1;
      prior_val = [(PriorDist.uniform(n_uniform,1)+PriorDist.uniform(n_uniform,1))/2 sqrt(((PriorDist.uniform(n_uniform,2)-PriorDist.uniform(n_uniform,1))^2)/12) PriorDist.uniform(n_uniform,:)];
   end;
   fprintf(fid,'%s\n',SepStr);
   if (strcmp(dist_str,'beta')==1)|(strcmp(dist_str,'uniform')==1);
      %
      % we have a beta or uniform prior
      %
      fprintf(fid,[PrtStr '    ' prt_val '    ' prt_val '\n'],add_str,ParameterNames.estimated(i,:),dist_add,dist_str,prior_val(1,1),theta(i),prior_val(1,2),prior_val(1,3),prior_val(1,4));
      fprintf(fid,'                                                               mean        st. dev.                                \n');
   elseif strcmp(dist_str,'normal')==1;
      %
      % we have a normal prior
      %
      fprintf(fid,[PrtStr '\n'],add_str,ParameterNames.estimated(i,:),dist_add,dist_str,theta(i),prior_val(1,1),prior_val(1,2));
      fprintf(fid,'                                                               mean        st. dev.                                \n');
   else;
      %
      % we have a gamma, inverse gamma, or left truncated normal prior
      %
      fprintf(fid,[PrtStr '    ' prt_val '\n'],add_str,ParameterNames.estimated(i,:),dist_add,dist_str,theta(i),prior_val(1,1),prior_val(1,2),prior_val(1,3));
      if strcmp(dist_str,'gamma')==1;
         fprintf(fid,'                                                               mean        st. dev.                                \n');
      elseif strcmp(dist_str,'invgamma')==1;;
         fprintf(fid,'                                                           location    deg. freedom                                \n');
      else;
         fprintf(fid,'                                                           location          spread                                \n');
      end;
   end;
end;
fprintf(fid,'%s\n\n',EndStr);
%
% Some information about the filename and its location
%
fprintf(fid,'Directory for file: %s\n',GetPath(TextFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(TextFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintPriorInformation.m
%
