function PrintPredictionEvent(file,VariableNames,PredEventData,YObsEventData,PredictionEvent,PredictionSampleStr,MaxPH,NumPaths,NumDraws,CurrINI)
% PrintPredictionEvent: Writes the prediction event results to file.
%
% USAGE:
%
%       PrintPredictionEvent(file,VariableNames,PredEventData,YObsEventData,PredictionEvent,PredictionSample,MaxPH,NumPaths,NumDraws,CurrINI)
%
% REQUIRED INPUT: file (string vector) with the name of the file that should be written.
%
%                 VariableNames (string matrix) with the names of the variables used for the
%                               prediction event calculations.
%
%                 PredEventData (n x 3) matrix where the first column holds the number of times
%                               the PredPaths data satisfies the event, the second hold the number
%                               of times the PredPaths data is greater than the event, while the
%                               third column holds the number of times the event has been tested.
%
%                 PredictionEvent (n x 3) matrix with the prediction event information. The first
%                               column holds the upper bound, the second the lower bound for the
%                               event, while the third holds the event length in periods.
%
%                 YObsEventData (n x 3) matrix where the first column hold the number of times
%                               the PredPaths (adjusted such that the mean is equal to the realized
%                               value) satisfies the event, the second holds the number of times
%                               the adjusted paths data is greater than the event, while the third
%                               column holds the number of tiems the event has been tested.
%
%                 PredictionSampleStr (string vector) with the prediction sample.
%
%                 MaxPH (integer) with the length of the prediction sample.
%
%                 NumPaths (integer) with the total number of prediction paths that were used.
%
%                 NumDraws (integer) with the toal number of parameters draws used for the compuations.
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: May 3, 2007.
%                        This version: January 15, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 4-5-2007:   Added the "NumDraws" input.
%
% * 6-6-2007:   The PredEventData now has the second column function.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 15-01-2009: Added code for the new input variable "YObsEventData".
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% some potential error catching
%
if isempty(PredictionEvent)==1;
   n = size(VariableNames,1);
   PredictionEvent = [Inf*ones(n,1) zeros(n,1) ones(n,1)];
end;
%
% open the file for printing
%
fid = fopen(file,'wt');
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*         P R E D I C T I O N   E V E N T   P R O B A B I L I T I E S          *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% some information about the prediction sample
%
fprintf(fid,'Prediction sample:                             %s\n',PredictionSampleStr);
fprintf(fid,'Prediction sample length:                      %0.0f\n',MaxPH);
fprintf(fid,'Number of prediction paths per parameter draw: %0.0f\n',NumPaths);
fprintf(fid,'Number of parameter draws used:                %0.0f\n\n',NumDraws);
%
if size(VariableNames,2)<20;
   SpStr = SpaceStr(20-size(VariableNames,2));
else;
   SpStr = '';
end;
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
prt_probval = ['%0.' num2str(CurrINI.decimals,'%0.0f') 'f'];
prt_str = ['%s%s      ' prt_probval '    >= ' prt_val ' and <= ' prt_val ' for %2.0f periods     %7.0f \n'];
prt_str2 = ['                          ' prt_probval '     < ' prt_val '                     for %2.0f periods\n'];
n = size(VariableNames,1);
fprintf(fid,'=========================================================================================================\n');
fprintf(fid,'            variable   probability    event                                              Number of tests\n');
for i=1:n;
   if i>1;
      fprintf(fid,'---------------------------------------------------------------------------------------------------------\n');
   else;
      fprintf(fid,'=========================================================================================================\n');
   end;
   fprintf(fid,prt_str,SpStr,VariableNames(i,:),PredEventData(i,1)/PredEventData(i,3),PredictionEvent(i,2),PredictionEvent(i,1),PredictionEvent(i,3),PredEventData(i,3));
   fprintf(fid,prt_str2,PredEventData(i,2)/PredEventData(i,3),PredictionEvent(i,2),PredictionEvent(i,3));
end;
fprintf(fid,'=========================================================================================================\n\n');
%
if isempty(YObsEventData)==0;
   fprintf(fid,'********************************************************************************\n');
   fprintf(fid,'*                                                                              *\n');
   fprintf(fid,'*         P R E D I C T I O N   E V E N T   P R O B A B I L I T I E S          *\n');
   fprintf(fid,'*           W H E N   P R E D I C T I V E   D I S T R I B U T I O N            *\n');
   fprintf(fid,'*      H A S   M E A N   E Q U A L   T O   R E A L I Z E D   V A L U E S       *\n');
   fprintf(fid,'*                                                                              *\n');
   fprintf(fid,'********************************************************************************\n\n');
   %
   fprintf(fid,'=========================================================================================================\n');
   fprintf(fid,'            variable   probability    event                                              Number of tests\n');
   for i=1:n;
      if i>1;
         fprintf(fid,'---------------------------------------------------------------------------------------------------------\n');
      else;
         fprintf(fid,'=========================================================================================================\n');
      end;
      fprintf(fid,prt_str,SpStr,VariableNames(i,:),YObsEventData(i,1)/YObsEventData(i,3),PredictionEvent(i,2),PredictionEvent(i,1),PredictionEvent(i,3),YObsEventData(i,3));
      fprintf(fid,prt_str2,YObsEventData(i,2)/YObsEventData(i,3),PredictionEvent(i,2),PredictionEvent(i,3));
   end;
   fprintf(fid,'=========================================================================================================\n\n');
end;
%
% bottom file data
%
fprintf(fid,'Directory for file: %s\n',GetPath(file));
fprintf(fid,'Name of file:       %s\n',GetFilename(file));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintPredictionEvent.m
%
