function PrintPosteriorSamplingSummary(DSGEModel,CurrINI,ParameterNames,PriorDist,thetaDist,ParData,thetaMargMode,UpdateParData,UpdateMargMode,UpdateNames,NumIter,NumBurnin,NumMeasureErrors,ScaleInitial,ScaleProp,PostSampleTime,AcceptRatio,LogMargsMH,CovProb,LogMargsCJ,StepLength,CurrChain,NumChains,thetaInit,RWvalue,MaxCorr)
% PrintPosteriorSamplingSummary: Writes summary information about the posterior distributions for the
%                                original parameters to file.
%
% USAGE:
%
%       PrintPosteriorSamplingSummary(DSGEModel,CurrINI,ParameterNames,PriorDist,thetaDist,ParData,...
%                                     thetaMargMode,UpdateParData,UpdateMargMode,UpdateNames,NumIter,...
%                                     NumBurnin,NumMeasureErrors,ScaleInitial,ScaleProp,PostSampleTime,...
%                                     AcceptRatio,LogMargsMH,CovProb,LogMargsCJ,StepLength,CurrChain,...
%                                     NumChains,thetaInit,RWvalue,MaxCorr)
%
% REQUIRED INPUT: DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this structure.
%
%                 CurrINI (structure) with initialization information.
%
%                 ParameterNames (structure) with fields "calibrated", "beta", "gamma", "normal", "invgamma", "truncnormal",
%                           "uniform", "all" and "estimated". Each field returns a string matrix with the parameter names.
%
%                 PriorDist (structure) with fields "beta", "gamma", "normal", "invgamma", and "truncnormal", where
%                           the values are matrices with the parameter values for the distributions. In the case of
%                           the normal, beta, and gamma these parameter values are the mean and the standard deviation.
%                           For the invgamma it is the s and q parameters, where q is a positive integer (degrees of
%                           freedom). Finally, for the left truncated normal there's a third parameter, the lower bound.
%
%                 thetaDist (vector) with integers 1 if gamma, 2 if beta, 3 if normal, 4 if invgamma, 5 if
%                           truncnormal, and 6 if uniform.
%
%                 ParData (structure), with fields "SampleMean" containing the sample
%                           mean of the different parameters, "SampleStd" with the sample
%                           standard deviations, and "SampleMedian" with the sample median
%                           parameter values. The field "SampleStdErr" contains numerical
%                           standard errors of the SampleMean estimate.
%
%                 thetaMargMode (vector) with the marginal posterior mode estimates of the original
%                           parameters.
%
%                 UpdateParData (structure) with the same fields as the ParData structure. The
%                           data is computed from the posterior values of the parameters that
%                           are a function of the estimated parameters.
%
%                 UpdateMargMode (vector) with the marginal posterior mode estimates of the
%                           parameters that are a function of the estimated parameters.
%
%                 UpdateNames (string matrix) with the names of the parameters that are a
%                           function of the estimated parameters.
%
%                 NumIter (integer) with the total number of posterior draws.
%
%                 NumBurnin (integer) with the number of burn-in draws for the posterior sampler.
%
%                 NumMeasureErrors (integer), giving the number of measurement errors in the model.
%
%                 ScaleInitial (non-negative real number), giving the scale factor for the initial phi value (c0)
%
%                 ScaleProp (positive real number), giving the scale factor for the proposal density (c).
%
%                 PostSampleTime (time) with the length it took the posterior sampler to finish from where
%                           it started.
%
%                 AcceptRatio (positive real number), giving the acceptance ratio for the posterior sampler.
%
%                 LogMargsMH (matrix) with sequential estimates of the log marginal likelihood using Geweke's
%                           modified harmonic mean estimator. First column is number of draws, while the remaining
%                           columns contain the estimated values for the CovProb values.
%
%                 CovProb (vector), with coverage probabilities.
%
%                 LogMargsCJ (matrix) with 3 columns. The first holds the number of draws used for the computation,
%                           the second the estimated log marginal likelihood based on the Chib & Jeliazkov estimator,
%                           while the third column gives the numerical standard error of the estimate.
%
%                 StepLength (positive scalar) with the step length that is used for the finite difference estimator
%                           of the inverse Hessian at the posterior mode.
%
%                 CurrChain (integer) with the number of the current MCMC chain.
%
%                 NumChains (integer) with the number of parallell MCMC chains.
%
%                 thetaInit (vector) with the initial values for the original parameters.
%
%                 RWvalue (scalar) with the weight on randomization when mutliple chains are used.
%
%                 MaxCorr (scalar) with the maximum absolute correlation scale factor.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: December 7, 2006.
%                        This version: February 26, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 08-12-2006: Finished the function.
%
% * 13-12-2006: Added printing of posterior distribution quantiles.
%
% * 10-04-2007: Took the variable "InverseHessianEstimator" into account. The new inputs are "StepLength", "CurrChain",
%               and "NumChains".
%
% * 11-04-2007: Added the inputs "phiInit" and "RWvalue".
%
% * 17-04-2007: Added the "MaxCorr" input.
%
% * 19-04-2007: Took the ParData field "SampleStdAuto" into account.
%
% * 13-11-2007: Updated the documentation.
%
% * 02-01-2008: Removed code using "DSGEModel.InitializeKalman".
%
% * 07-01-2008: Edited the header for the output file.
%
% * 23-05-2008: Updated the documentation.
%
% * 22-07-2008: Took into account that the gamma and inverse gamma distributions now use
%               the lower bound parameter.
%
% * 23-07-2008: Took into account that the beta distribution supports general lower and upper bounds.
%
% * 23-10-2008: Added information about the number of parameters to estimate and the total number of parameters.
%
% * 21-01-2009: Updated the documentation.
%
% * 26-02-2009: Changed the name of the input variable "thetaMode" to "thetaMargMode", reflecting that
%               it holds the values for the mode of the marginal posterior. The input variables "UpdateParData",
%               "UpdateMargMode", and "UpdateNames" were added, as well code for writing to file the information
%               about the parameters that are a function of the estimated parameters.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

NumParam = length(ParData.SampleMean);
NumAdditional = length(UpdateMargMode);
PostSampleTextFile = [DSGEModel.OutputDirectory '\PosteriorSamplingSummary-' DSGEModel.NameOfModel '-' num2str(DSGEModel.InverseHessianEstimator,'%0.0f') '-' num2str(CurrChain,'%0.0f') '.' num2str(NumChains,'%0.0f') '-' num2str(NumIter,'%0.0f') '.txt'];
%
fid = fopen(PostSampleTextFile,'wt');
%
% the header information
%
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*     P O S T E R I O R   S A M P L I N G   W I T H   Y A D A   U S I N G      *\n');
fprintf(fid,'*   T H E   R A N D O M - W A L K   M E T R O P O L I S   A L G O R I T H M    *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% Model info
%
fprintf(fid,'Name of model:                    %s\n',DSGEModel.NameOfModel);
fprintf(fid,'Data construction file:           %s\n',GetFilename(DSGEModel.DataConstructionFile));
fprintf(fid,'Measurement equation file:        %s\n',GetFilename(DSGEModel.MeasurementEquationFile));
fprintf(fid,'Prior distribution data file:     %s\n',GetFilename(DSGEModel.PriorFile));
fprintf(fid,'Sheet name:                       %s\n',DSGEModel.PriorFileSheet);
if FileExist(DSGEModel.UpdateParameterFile)==1;
   fprintf(fid,'Update parameters file:           %s\n',GetFilename(DSGEModel.UpdateParameterFile));
end;
if FileExist(DSGEModel.InitializeParameterFile)==1;
   fprintf(fid,'Initialize parameters file:       %s\n',GetFilename(DSGEModel.InitializeParameterFile));
end;
fprintf(fid,'AiM model file:                   %s\n\n',GetFilename(DSGEModel.AIMFile));
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*              S A M P L E   &   D A T A   I N F O R M A T I O N               *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% Sample information
%
[NewStartYear,NewStartPeriod] = AdjustSampleStart(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.DataFrequency,DSGEModel.KalmanFirstObservation-1);
%
fprintf(fid,'Selected sample:                  %s:%s -- %s:%s\n',DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.SubEndYear,DSGEModel.SubEndPeriod);
fprintf(fid,'Number of observations:           %0.0f\n',DSGEModel.LastPeriod-DSGEModel.FirstPeriod+1);
fprintf(fid,'Number of observations used to\n');
fprintf(fid,'initialize the Kalman filter:     %0.0f\n',DSGEModel.KalmanFirstObservation-1);
fprintf(fid,'Effective number of observations: %0.0f\n',DSGEModel.LastPeriod-DSGEModel.FirstPeriod+1-(DSGEModel.KalmanFirstObservation-1));
fprintf(fid,'Estimation sample:                %s:%s -- %s:%s\n',NewStartYear,NewStartPeriod,DSGEModel.SubEndYear,DSGEModel.SubEndPeriod);
%
fprintf(fid,'Number of observed variables:     %0.0f\n',min(size(DSGEModel.Y)));
fprintf(fid,'Number of exogenous variables:    %0.0f\n',min(size(DSGEModel.X)));
fprintf(fid,'Number of state variables:        %0.0f\n',size(DSGEModel.StateVariableNames,1));
fprintf(fid,'Number of state shocks:           %0.0f\n',size(DSGEModel.StateShockNames,1));
fprintf(fid,'Number of measurement errors:     %0.0f\n\n',NumMeasureErrors);
%
% observed variable names, state variable names and state shocks
%
VarNames = StrWrap(vecstr(DSGEModel.VariableNames,' '),40);
fprintf(fid,'Names of observed variables:      %s\n',VarNames(1,:));
for i=2:size(VarNames,1);
   if strcmp(StringTrim(VarNames(i,:)),'')==0;
      fprintf(fid,'                                  %s\n',VarNames(i,:));
   end;
end;
if min(size(DSGEModel.X))>0;
   XVarNames = StrWrap(vecstr(DSGEModel.XVariableNames,' '),40);
   fprintf(fid,'Names of exogenous variables:     %s\n',XVarNames(1,:));
   for i=2:size(XVarNames,1);
      if strcmp(StringTrim(XVarNames(i,:)),'')==0;
         fprintf(fid,'                                  %s\n',XVarNames(i,:));
      end;
   end;
end;
StateVarNames = StrWrap(vecstr(DSGEModel.StateVariableNames,' '),40);
fprintf(fid,'Names of state variables:         %s\n',StateVarNames(1,:));
for i=2:size(StateVarNames,1);
   if strcmp(StringTrim(StateVarNames(i,:)),'')==0;
      fprintf(fid,'                                  %s\n',StateVarNames(i,:));
   end;
end;
StateShockNames = StrWrap(vecstr(DSGEModel.StateShockNames,' '),40);
fprintf(fid,'Names of the state shocks:        %s\n',StateShockNames(1,:));
for i=2:size(StateShockNames,1);
   if strcmp(StringTrim(StateShockNames(i,:)),'')==0;
      fprintf(fid,'                                  %s\n',StateShockNames(i,:));
   end;
end;
%
DivStr = MultiCharStr('=',80);
%
% posterior sampler information
%
fprintf(fid,'\n********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*            P O S T E R I O R   D R A W S   I N F O R M A T I O N             *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
fprintf(fid,'Number of posterior draws:             %0.0f\n',NumIter);
fprintf(fid,'Number of burn-in draws:               %0.0f\n',NumBurnin);
fprintf(fid,'MCMC chain number:                     %0.0f\n',CurrChain);
if DSGEModel.InverseHessianEstimator==1;
   InvHessianTxt = 'Optimization routine';
elseif DSGEModel.InverseHessianEstimator==2;
   InvHessianTxt = 'Quadratic modification';
else;
   InvHessianTxt = 'Finite difference';
end;
fprintf(fid,'Inverse Hessian estimator:             %s\n',InvHessianTxt);
if DSGEModel.InverseHessianEstimator==3;
   fprintf(fid,'Step length:                           %s\n',StepLength);
end;
fprintf(fid,'Max absolute corelation scale factor:  %0.4f\n',MaxCorr);
if NumChains>1;
   fprintf(fid,'Weight on randomization:               %0.4f\n',RWvalue);
else;
   fprintf(fid,'Scale factor for initial phi density:  %0.4f\n',ScaleInitial);
end;
fprintf(fid,'Scale factor for proposal phi density: %0.4f\n',ScaleProp);
fprintf(fid,'Posterior sampling time:               %s (hrs:min:sec)\n',SecondsToHoursStr(PostSampleTime));
fprintf(fid,'Acceptance ratio:                      %0.4f\n\n',AcceptRatio);
if DSGEModel.MarginalLikelihoodValue~=4;
   prt_val = ['%' num2str(9+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
   %
   % write results for log marginal likelihoods
   %
   fprintf(fid,'********************************************************************************\n');
   fprintf(fid,'*                                                                              *\n');
   fprintf(fid,'*         M A R G I N A L   L I K E L I H O O D   E S T I M A T I O N          *\n');
   fprintf(fid,'*                                                                              *\n');
   fprintf(fid,'********************************************************************************\n\n');
   if isempty(LogMargsMH)==0;
      %
      % Geweke's modified harmonic mean estimator
      %
      prt_str = 'Modified Harmonic Mean';
      top_str = '                      ';
      for i=1:length(CovProb);
         top_str = [top_str '        ' num2str(CovProb(i)*100,'%0.0f') ' percent'];
         prt_str = [prt_str '   ' prt_val];
      end;
      TopStr = MultiCharStr('=',length(top_str));
      MidStr = MultiCharStr('-',length(top_str));
      fprintf(fid,'%s\n',TopStr);
      fprintf(fid,'                              Coverage probabilities for the estimator\n');
      fprintf(fid,[top_str '\n']);
      fprintf(fid,'%s\n',MidStr);
      fprintf(fid,[prt_str '\n'],LogMargsMH(size(LogMargsMH,1),2:size(LogMargsMH,2))');
      fprintf(fid,'%s\n\n',TopStr);
   end;
   if isempty(LogMargsCJ)==0;
      fprintf(fid,'==========================================================\n');
      fprintf(fid,'                          point estimate        std. error\n');
      fprintf(fid,'----------------------------------------------------------\n');
      fprintf(fid,['Chib & Jeliazkov         ' prt_val '   ' prt_val '\n'],LogMargsCJ(size(LogMargsCJ,1),2),LogMargsCJ(size(LogMargsCJ,1),3));
      fprintf(fid,'==========================================================\n\n');
   end;
end;
%
% deal with the parameters
%
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*                             P A R A M E T E R S                              *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
fprintf(fid,'Number of parameters to estimate: %0.0f\n',length(thetaMargMode));
if NumAdditional>0;
   fprintf(fid,'Number of update parameters:      %0.0f\n',NumAdditional);
end;
if size(ParameterNames.calibrated,1)>0;
   fprintf(fid,'Number of calibrated parameters:  %0.0f\n',size(ParameterNames.calibrated,1));
end;
fprintf(fid,'Total number of parameters:       %0.0f\n\n',size(ParameterNames.all,1));
%
ns = size(ParameterNames.estimated,2);
if ns<20;
   add_str = SpaceStr(20-ns);
else;
   add_str = '';
end;
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
prt_str = ['%s%s   ' prt_val '   ' prt_val '   ' prt_val '   |   ' prt_val '   ' prt_val '   ' prt_val '   ' prt_val];
NewDivStr = MultiCharStr('-',129);
fprintf(fid,'=================================================================================================================================\n');
fprintf(fid,'                                                       Original Parameters\n');
fprintf(fid,'           parameter       mean          median          mode       |      st. dev.   auto st. dev.       RNE             IF\n');
fprintf(fid,'                     [std err of mean]\n');
fprintf(fid,'                       initial value\n');
if (isempty(PriorDist.truncnormal)==1)&(isempty(PriorDist.invgamma)==1)&(isempty(PriorDist.gamma)==1)&(isempty(PriorDist.beta)==1);
   fprintf(fid,'         (prior dist    parameter 1    parameter 2)\n');
else;
   if isempty(PriorDist.beta)==1;
      fprintf(fid,'         (prior dist    parameter 1    parameter 2    parameter 3)\n');
   else;
      fprintf(fid,'         (prior dist    parameter 1    parameter 2    parameter 3\n');
      fprintf(fid,'                                                      parameter 4)\n');
   end;
end;
%
n_gamma = 0;
n_beta = 0;
n_normal = 0;
n_invgamma = 0;
n_truncnormal = 0;
n_uniform = 0;
for i=1:NumParam;
   if i>1;
      fprintf(fid,'%s\n',NewDivStr);
   else;
      fprintf(fid,'=================================================================================================================================\n');
   end;
   if thetaDist(i)==1;
      dist_str = 'gamma';
      dist_add = SpaceStr(6);
      n_gamma = n_gamma+1;
      prior_val = PriorDist.gamma(n_gamma,:);
   elseif thetaDist(i)==2;
      dist_str = 'beta';
      dist_add = SpaceStr(7);
      n_beta = n_beta+1;
      prior_val = PriorDist.beta(n_beta,:);
   elseif thetaDist(i)==3;
      dist_str = 'normal';
      dist_add = SpaceStr(5);
      n_normal = n_normal+1;
      prior_val = PriorDist.normal(n_normal,:);
   elseif thetaDist(i)==4;
      dist_str = 'invgamma';
      dist_add = SpaceStr(3);
      n_invgamma = n_invgamma+1;
      prior_val = PriorDist.invgamma(n_invgamma,:);
   elseif thetaDist(i)==5;
      dist_str = 'truncnormal';
      dist_add = '';
      n_truncnormal = n_truncnormal+1;
      prior_val = PriorDist.truncnormal(n_truncnormal,:);
   elseif thetaDist(i)==6;
      dist_str = 'uniform';
      dist_add = SpaceStr(4);
      n_uniform = n_uniform+1;
      prior_val = PriorDist.uniform(n_uniform,:);
   end;
   fprintf(fid,[prt_str '\n'],add_str,ParameterNames.estimated(i,:),ParData.SampleMean(i),ParData.SampleMedian(i),thetaMargMode(i),ParData.SampleStd(i),ParData.SampleStdAuto(i),(ParData.SampleStd(i)/ParData.SampleStdAuto(i))^2,(ParData.SampleStdAuto(i)/ParData.SampleStd(i))^2);
   fprintf(fid,['                      [' prt_val ']\n'],ParData.SampleStdErr(i));
   fprintf(fid,['                       ' prt_val '\n'],thetaInit(i));
   if (strcmp(dist_str,'truncnormal')==0)&(strcmp(dist_str,'invgamma')==0)&(strcmp(dist_str,'gamma')==0)&(strcmp(dist_str,'beta')==0);
      fprintf(fid,['        %s(%s   ' prt_val '   ' prt_val ')\n'],dist_add,dist_str,prior_val(1,1),prior_val(1,2));
   else;
      if strcmp(dist_str,'beta')==0;
         fprintf(fid,['        %s(%s   ' prt_val '   ' prt_val '   ' prt_val ')\n'],dist_add,dist_str,prior_val(1,1),prior_val(1,2),prior_val(1,3));
      else;
         fprintf(fid,['        %s(%s   ' prt_val '   ' prt_val '   ' prt_val '\n'],dist_add,dist_str,prior_val(1,1),prior_val(1,2),prior_val(1,3));
         fprintf(fid,['                                                     ' prt_val ')\n'],prior_val(1,4));
      end;
   end;
end;
fprintf(fid,'=================================================================================================================================\n\n');
%
% check if we have additional parameters
%
if isempty(UpdateMargMode)==0;
   nns = size(UpdateNames,2);
   if nns<20;
      AddUpdStr = SpaceStr(20-nns);
   else;
      AddUpdStr = '';
   end;
   fprintf(fid,'=================================================================================================================================\n');
   fprintf(fid,'                                                       Update Parameters\n');
   fprintf(fid,'           parameter       mean          median          mode       |      st. dev.   auto st. dev.       RNE             IF\n');
   fprintf(fid,'                     [std err of mean]\n');
   for i=1:NumAdditional;
      if i>1;
         fprintf(fid,'%s\n',NewDivStr);
      else;
         fprintf(fid,'=================================================================================================================================\n');
      end;
      fprintf(fid,[prt_str '\n'],AddUpdStr,UpdateNames(i,:),UpdateParData.SampleMean(i),UpdateParData.SampleMedian(i),UpdateMargMode(i),UpdateParData.SampleStd(i),UpdateParData.SampleStdAuto(i),(UpdateParData.SampleStd(i)/UpdateParData.SampleStdAuto(i))^2,(UpdateParData.SampleStdAuto(i)/UpdateParData.SampleStd(i))^2);
      fprintf(fid,['                      [' prt_val ']\n'],UpdateParData.SampleStdErr(i));
   end;
   fprintf(fid,'=================================================================================================================================\n\n');
end;
%
fprintf(fid,'NOTE: RNE measures the relative numerical efficiency. It is defined as the ratio between the variance\n');
fprintf(fid,'      of the draws under the assumption that they are iid and the variance of the draws when autocorrelation\n');
fprintf(fid,'      is taken into account. This value is equal to the ratio between the previous two columns squared. The\n');
fprintf(fid,'      inverse of the RNE is sometimes called the inefficiency factor (IF) and is equal to 1+2*sum(1,K) rho(k),\n');
fprintf(fid,'      where rho(k) is the autocorrelation at lag k, and K is %0.0f.\n\n',floor((NumIter-NumBurnin)^(1/2.01)));
%
% print parameter quantiles
%
prt_str = ['%s%s   ' prt_val '   ' prt_val '   ' prt_val '   ' prt_val '   ' prt_val '   ' prt_val '   ' prt_val '   ' prt_val '   ' prt_val];
fprintf(fid,'===========================================================================================================================================================\n');
fprintf(fid,'                                                      Posterior Quantiles for the Original Parameters                                                      \n');
fprintf(fid,'           parameter        2.5%%           5%%            10%%            20%%            50%%            80%%            90%%            95%%            97.5%%\n');
for i=1:NumParam;
   fprintf(fid,'-----------------------------------------------------------------------------------------------------------------------------------------------------------\n');
   fprintf(fid,[prt_str '\n'],add_str,ParameterNames.estimated(i,:),ParData.Quantiles(i,:)');
end;
fprintf(fid,'===========================================================================================================================================================\n\n');
%
if isempty(UpdateMargMode)==0;
   fprintf(fid,'===========================================================================================================================================================\n');
   fprintf(fid,'                                                      Posterior Quantiles for the Update Parameters                                                      \n');
   fprintf(fid,'           parameter        2.5%%           5%%            10%%            20%%            50%%            80%%            90%%            95%%            97.5%%\n');
   for i=1:NumAdditional;
      fprintf(fid,'-----------------------------------------------------------------------------------------------------------------------------------------------------------\n');
      fprintf(fid,[prt_str '\n'],AddUpdStr,UpdateNames(i,:),UpdateParData.Quantiles(i,:)');
   end;
   fprintf(fid,'===========================================================================================================================================================\n\n');
end;
%
% Some information about the filename and its location
%
fprintf(fid,'Directory for file: %s\n',GetPath(PostSampleTextFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(PostSampleTextFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintPosteriorSamplingSummary.m
%
