function PrintPosteriorModeSummary(DSGEModel,CurrINI,ParameterNames,thetaMode,theta,thetaDist,PriorDist,ThetaPhiPartial,InverseHessian,phiFittedStd,LogPostDensity,LogLikeDensity,LogPriorDensity,LogJacobianValue,ExtremeCorrParameters,OptAlgorithm,MaxTime,ToleranceValue,MaxIter,NumIter,FuncDiff,MaxEigenvalueIV,MaxEigenvalueMode,LogPosteriorPhiIV,LogPosteriorThetaIV,LogPriorDensityIV,NumMeasureErrors,MFEYMode,MSEYMode,MFEYIV,MSEYIV)
% PrintPosteriorModeSummary: Writes a short summary of the output from sucessful posterior mode estimation run to text file
%                            in the output directory.
%
% USAGE:
%
%       PrintPosteriorModeSummary(DSGEModel,CurrINI,ParameterNames,thetaMode,theta, ...
%                 thetaDist,PriorDist,ThetaPhiPartial,InverseHessian,phiFittedStd,LogPostDensity, ...
%                 LogLikeDensity,LogPriorDensity,LogJacobianValue,ExtremeCorrParameters, ...
%                 OptAlgorithm,MaxTime,ToleranceValue,MaxIter,NumIter,FuncDiff,MaxEigenvalueIV, ...
%                 MaxEigenvalueMode,LogPosteriorPhiIV,LogPosteriorThetaIV,LogPriorDensityIV, ...
%                 NumMeasureErrors,MFEYMode,MSEYMode,MFEYIV,MSEYIV)
%
% REQUIRED INPUT: DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this structure.
%
%                 CurrINI (structure), with initialization information.
%
%                 ParameterNames (structure) with fields "calibrated", "beta", "gamma", "normal", "invgamma", "truncnormal",
%                           "uniform", "all" and "estimated". Each field returns a string matrix with the parameter names.
%
%                 thetaMode (vector) with the original parameters to estimate at the posterior mode
%
%                 theta (vector) with the initial values of the original parameters.
%
%                 thetaDist (vector) with integers 1 if gamma, 2 if beta, 3 if normal, 4 if invgamma, 5 if
%                           truncnormal, and 6 if uniform.
%
%                 PriorDist (structure) with fields "beta", "beta_ab", "gamma", "gamma_ab", "normal", "invgamma",
%                           "truncnormal", and "uniform" where the values are matrices with the parameter values for the
%                           distributions. In the case of the normal, beta, and gamma these parameter values are
%                           the mean and the standard deviation. For the invgamma it is the s and q parameters,
%                           where q is a positive integer (degrees of freedom). For the left truncated normal
%                           there's a third parameter, the lower bound. Finally, for the uniform the parameters
%                           are the lower and the upper bound.
%
%                 ThetaPhiPartial (matrix) where the diagonal entries give the partial derivatives of the original
%                           parameters wrt the transformed parameters. Evalues at the mode.
%
%                 InverseHessian (matrix), the inverse of the Hessian at the mode of the transformed parameters.
%                           This matrix is provides directly from the optimization routine.
%
%                 phiFittedStd (vector) of standard deviations of the transformed parameters. This vector is given
%                           as output by "VillaniCheckOptimum".
%
%                 LogPostDensity (scalar), the value of the log posterior of the transformed parameters at the mode.
%
%                 LogLikeDensity (scalar), the value of the log likelihood function of the original parameters at the mode.
%
%                 LogPriorDensity (scalar), the value of the log prior density of the original parameters at the mode.
%
%                 LogJacobianValue (scalar), the value of the log Jacobian in the transformation from the original
%                            parameters to the transformed parameters.
%
%                 ExtremeCorrParameters (cell array), with parameter pairs having absolute correlations equal to or
%                            greater than 0.7. The dimension of the cell array is equal to the number of such extreme
%                            correlations times 3. The columns contains the name of parameter 1 (string), the name of
%                            parameter 2 (string), and the estimated correlation (scalar).
%
%                 OptAlgorithm (string vector), with either "fminunc" or "csminwel" being the optimization algorithm
%                            used to calculate the posterior mode.
%
%                 MaxTime (scalar), the time in seconds to complete the optimization problem.
%
%                 ToleranceValue (scalar), the tolerance value used when evaluating if the optimization algorithm has
%                            converged or not.
%
%                 MaxIter (positive integer), the maximum number of iterations considered by the optimization algorithm.
%
%                 NumIter (positive integer), the actual number of iteration used by the optimization algorithm.
%
%                 FuncDiff (scalar), the value of the convergence measure.
%
%                 MaxEigenvalueIV (scalar), the largest absolute eigenvalue of the state equation transition matrix at
%                            the initial value of theta.
%
%                 MaxEigenvalueMode (scalar), the largest absolute eigenvalue of the state equation transition matrix at
%                            the posterior mode of theta.
%
%                 LogPosteriorPhiIV (scalar), value of the log posterior of phi at the initial value of phi.
%
%                 LogPosteriorThetaIV (scalar), value of the log posterior of theta at the initial value of theta.
%
%                 LogPriorDensityIV (sclara), value of the log prior desnity of theta at the initial value of theta.
%
%                 NumMeasureErrors (integer), giving the number of measurement errors in the model.
%
%                 MFEYMode (vector), with the mean forecast errors of the 1-step ahead forecasts of the observed
%                            variables when the parameters are evaluated at the mode.
%
%                 MSEYMode (matrix) with the mean squared errors of the 1-step ahead forecasts of the observed
%                            variables when the parameters are evaluated at the mode.
%
%                 MFEYIV (vector), with the mean forecast errors of the 1-step ahead forecasts of the observed
%                            variables when the parameters are evaluated at the initial values.
%
%                 MSEYIV (matrix) with the mean squared errors of the 1-step ahead forecasts of the observed
%                            variables when the parameters are evaluated at the initial values.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: October 20, 2006.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 2-11-2006:  Made sure that Parameter 3 is set to NaN when the prior only depends on 2 parameters
%               but output requires 3 parameters.
%
% * 3-11-2006:  Cosmetic fixes while enjoying "The truth will set you free" by TFK.
%
% * 9-11-2006:  Added information about the actual sample used.
%
% * 20-11-2006: Changed the prt_val string when the log posterior values are written to file from 12.6f to 15.6f.
%
% * 22-11-2006: Fixed a spelling error.
%
% * 7-12-2006:  Updated the documentation.
%
% * 8-12-2006:  Added "(hrs:min:sec)" to the time output string.
%
% * 17-4-2007:  Updated the documentation.
%
% * 13-11-2007: Updated the documentation.
%
% * 7-12-2007:  Checking a bug with StrWrap. Bug found and fixed!
%
% * 2-1-2008:   Removed code using "DSGEModel.InitializeKalman".
%
% * 23-5-2008:  Updated the documentation.
%
% * 22-7-2008:  Took into account that the gamma and inverse gamma distributions now use
%               the lower bound parameter.
%
% * 23-7-2008:  Took into account that the beta distribution supports general lower and upper bounds.
%
% * 23-10-2008: Added information about the number of parameters to estimate and the total number of parameters.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

ModeTextFile = [DSGEModel.OutputDirectory '\PosteriorModeSummary-' DSGEModel.NameOfModel '.txt'];
%
fid = fopen(ModeTextFile,'wt');
%
% the header information
%
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*    P O S T E R I O R   M O D E   E S T I M A T I O N   W I T H   Y A D A     *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% Model info
%
fprintf(fid,'Name of model:                    %s\n',DSGEModel.NameOfModel);
fprintf(fid,'Data construction file:           %s\n',GetFilename(DSGEModel.DataConstructionFile));
fprintf(fid,'Measurement equation file:        %s\n',GetFilename(DSGEModel.MeasurementEquationFile));
fprintf(fid,'Prior distribution data file:     %s\n',GetFilename(DSGEModel.PriorFile));
fprintf(fid,'Sheet name:                       %s\n',DSGEModel.PriorFileSheet);
if FileExist(DSGEModel.UpdateParameterFile)==1;
   fprintf(fid,'Update parameters file:           %s\n',GetFilename(DSGEModel.UpdateParameterFile));
end;
if FileExist(DSGEModel.InitializeParameterFile)==1;
   fprintf(fid,'Initialize parameters file:       %s\n',GetFilename(DSGEModel.InitializeParameterFile));
end;
fprintf(fid,'AiM model file:                   %s\n\n',GetFilename(DSGEModel.AIMFile));
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*              S A M P L E   &   D A T A   I N F O R M A T I O N               *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% Sample information
%
[NewStartYear,NewStartPeriod] = AdjustSampleStart(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.DataFrequency,DSGEModel.KalmanFirstObservation-1);
%
fprintf(fid,'Selected sample:                  %s:%s -- %s:%s\n',DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.SubEndYear,DSGEModel.SubEndPeriod);
fprintf(fid,'Number of observations:           %0.0f\n',DSGEModel.LastPeriod-DSGEModel.FirstPeriod+1);
fprintf(fid,'Number of observations used to\n');
fprintf(fid,'initialize the Kalman filter:     %0.0f\n',DSGEModel.KalmanFirstObservation-1);
fprintf(fid,'Effective number of observations: %0.0f\n',DSGEModel.LastPeriod-DSGEModel.FirstPeriod+1-(DSGEModel.KalmanFirstObservation-1));
fprintf(fid,'Estimation sample:                %s:%s -- %s:%s\n',NewStartYear,NewStartPeriod,DSGEModel.SubEndYear,DSGEModel.SubEndPeriod);
%
fprintf(fid,'Number of observed variables:     %0.0f\n',min(size(DSGEModel.Y)));
fprintf(fid,'Number of exogenous variables:    %0.0f\n',min(size(DSGEModel.X)));
fprintf(fid,'Number of state variables:        %0.0f\n',size(DSGEModel.StateVariableNames,1));
fprintf(fid,'Number of state shocks:           %0.0f\n',size(DSGEModel.StateShockNames,1));
fprintf(fid,'Number of measurement errors:     %0.0f\n\n',NumMeasureErrors);
%
% observed variable names, state variable names and state shocks
%
VarNames = StrWrap(vecstr(DSGEModel.VariableNames,' '),40);
fprintf(fid,'Names of observed variables:      %s\n',VarNames(1,:));
for i=2:size(VarNames,1);
   if strcmp(StringTrim(VarNames(i,:)),'')==0;
      fprintf(fid,'                                  %s\n',VarNames(i,:));
   end;
end;
if min(size(DSGEModel.X))>0;
   XVarNames = StrWrap(vecstr(DSGEModel.XVariableNames,' '),40);
   fprintf(fid,'Names of exogenous variables:     %s\n',XVarNames(1,:));
   for i=2:size(XVarNames,1);
      if strcmp(StringTrim(XVarNames(i,:)),'')==0;
         fprintf(fid,'                                  %s\n',XVarNames(i,:));
      end;
   end;
end;
StateVarNames = StrWrap(vecstr(DSGEModel.StateVariableNames,' '),40);
fprintf(fid,'Names of state variables:         %s\n',StateVarNames(1,:));
for i=2:size(StateVarNames,1);
   if strcmp(StringTrim(StateVarNames(i,:)),'')==0;
      fprintf(fid,'                                  %s\n',StateVarNames(i,:));
   end;
end;
StateShockNames = StrWrap(vecstr(DSGEModel.StateShockNames,' '),40);
fprintf(fid,'Names of the state shocks:        %s\n',StateShockNames(1,:));
for i=2:size(StateShockNames,1);
   if strcmp(StringTrim(StateShockNames(i,:)),'')==0;
      fprintf(fid,'                                  %s\n',StateShockNames(i,:));
   end;
end;
%
fprintf(fid,'\n********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*               O P T I M I Z A T I O N   I N F O R M A T I O N                *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% optimization information
%
fprintf(fid,'Optimization algorithm:           %s\n',OptAlgorithm);
fprintf(fid,'Time:                             %s (hrs:min:sec)\n',SecondsToHoursStr(MaxTime));
fprintf(fid,'Tolerance level:                  %0.10f\n',ToleranceValue);
fprintf(fid,'Maximum number of iterations:     %0.0f\n',MaxIter);
fprintf(fid,'Number of iterations:             %0.0f\n',NumIter);
fprintf(fid,'Convergence value:                %0.10f\n\n',FuncDiff);
%
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*              P O S T E R I O R   D E N S I T Y   R E S U L T S               *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% Log posterior data
%
LogMargLike = MargLikeLaplace(LogPostDensity,InverseHessian);
%
prt_val = ['%' num2str(9+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
fprintf(fid,'--------------------------------------------------------------------------\n');
fprintf(fid,'|                     Log Posterior Density at Mode                      |\n');
fprintf(fid,'|------------------------------------------------------------------------|\n');
fprintf(fid,['|  Value of the log posterior density of phi at mode:   ' prt_val '  |\n'],LogPostDensity);
fprintf(fid,['|  Value of the log posterior density of theta at mode: ' prt_val '  |\n'],LogPostDensity-LogJacobianValue);
fprintf(fid,['|  Value of the log likelihood at theta mode:           ' prt_val '  |\n'],LogLikeDensity);
fprintf(fid,['|  Value of the log prior density at theta mode:        ' prt_val '  |\n'],LogPriorDensity);
fprintf(fid,['|  Value of the log Jacobian (dtheta/dphi) at mode:     ' prt_val '  |\n'],LogJacobianValue);
fprintf(fid,'|                                                                        |\n');
fprintf(fid,'|  Value of the Laplace approximation of the log marginal                |\n');
fprintf(fid,['|  likelihood based on the posterior mode of phi:       ' prt_val '  |\n'],LogMargLike);
fprintf(fid,'|------------------------------------------------------------------------|\n');
fprintf(fid,'|                 Log Posterior Density at Initial Value                 |\n');
fprintf(fid,'|------------------------------------------------------------------------|\n');
fprintf(fid,['|  Initial value of the log posterior density of phi:   ' prt_val '  |\n'],LogPosteriorPhiIV);
fprintf(fid,['|  Initial value of the log posterior density of theta: ' prt_val '  |\n'],LogPosteriorThetaIV);
fprintf(fid,['|  Initial value of the log likelihood at theta:        ' prt_val '  |\n'],LogPosteriorThetaIV-LogPriorDensityIV);
fprintf(fid,['|  Initial value of the log prior density of theta:     ' prt_val '  |\n'],LogPriorDensityIV);
fprintf(fid,['|  Initial value of the log Jacobian (dtheta/dphi):     ' prt_val '  |\n'],LogPosteriorPhiIV-LogPosteriorThetaIV);
fprintf(fid,'--------------------------------------------------------------------------\n\n');
fprintf(fid,'NOTE: log posterior density of phi = (log likelihood at theta + \n');
fprintf(fid,'      log prior density at theta + log Jacobian), while the log\n');
fprintf(fid,'      posterior density of theta = (log posterior density of phi -\n');
fprintf(fid,'      log Jacobian).\n\n');
%
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*                             P A R A M E T E R S                              *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% The parameter estimates
%
fprintf(fid,'Number of parameters to estimate: %0.0f\n',length(theta));
fprintf(fid,'Total number of parameters:       %0.0f\n\n',size(ParameterNames.all,1));
%
% check the length of parameter names string
%
ns = size(ParameterNames.estimated,2);
if ns<20;
   add_str = SpaceStr(20-ns);
else;
   add_str = '';
end;
prt_str_1 = ['%s%s    ' prt_val '    ' prt_val '    ' prt_val];
prt_str_2 = ['        %s(%s    ' prt_val '    ' prt_val];
%
% Calculate an approximation of the inverse Hessian for theta, the original parameters, using the
% inverse Hessian at the mode for phi, the transformed parameters, and the partial derivatives of
% theta wrt phi
%
thetaInverseHessian = ThetaPhiPartial*InverseHessian*ThetaPhiPartial;
%
fprintf(fid,'============================================================================\n');
fprintf(fid,'                            Original Parameters                             \n');
fprintf(fid,'      parameter          theta mode        st dev         init val\n');
if (isempty(PriorDist.truncnormal)==1)&(isempty(PriorDist.invgamma)==1)&(isempty(PriorDist.gamma)==1)&(isempty(PriorDist.beta)==1);
   fprintf(fid,'     (prior dist        parameter 1     parameter 2)\n');
else;
   if isempty(PriorDist.beta)==1;
      fprintf(fid,'     (prior dist        parameter 1     parameter 2     parameter 3)\n');
   else;
      fprintf(fid,'     (prior dist        parameter 1     parameter 2     parameter 3\n');
      fprintf(fid,'                                                        parameter 4)\n');
   end;
end;
%
n_gamma = 0;
n_beta = 0;
n_normal = 0;
n_invgamma = 0;
n_truncnormal = 0;
n_uniform = 0;
for i=1:length(thetaMode);
   if thetaDist(i)==1;
      dist_str = 'gamma';
      dist_add = SpaceStr(6);
      n_gamma = n_gamma+1;
      prior_val = PriorDist.gamma(n_gamma,:);
   elseif thetaDist(i)==2;
      dist_str = 'beta';
      dist_add = SpaceStr(7);
      n_beta = n_beta+1;
      prior_val = PriorDist.beta(n_beta,:);
   elseif thetaDist(i)==3;
      dist_str = 'normal';
      dist_add = SpaceStr(5);
      n_normal = n_normal+1;
      prior_val = PriorDist.normal(n_normal,:);
   elseif thetaDist(i)==4;
      dist_str = 'invgamma';
      dist_add = SpaceStr(3);
      n_invgamma = n_invgamma+1;
      prior_val = PriorDist.invgamma(n_invgamma,:);
   elseif thetaDist(i)==5;
      dist_str = 'truncnormal';
      dist_add = '';
      n_truncnormal = n_truncnormal+1;
      prior_val = PriorDist.truncnormal(n_truncnormal,:);
   elseif thetaDist(i)==6;
      dist_str = 'uniform';
      dist_add = SpaceStr(4);
      n_uniform = n_uniform+1;
      prior_val = PriorDist.uniform(n_uniform,:);
   end;
   fprintf(fid,'----------------------------------------------------------------------------\n');
   fprintf(fid,[prt_str_1 '\n'],add_str,ParameterNames.estimated(i,:),thetaMode(i),sqrt(thetaInverseHessian(i,i)),theta(i));
   if max(isnan(phiFittedStd))==0;
      fprintf(fid,['                                        ' prt_val '\n'],ThetaPhiPartial(i,i)*phiFittedStd(i));
   end;
   if (strcmp(dist_str,'truncnormal')==0)&(strcmp(dist_str,'invgamma')==0)&(strcmp(dist_str,'gamma')==0)&(strcmp(dist_str,'beta')==0);
      fprintf(fid,[prt_str_2 ')\n'],dist_add,dist_str,prior_val(1,1),prior_val(1,2));
   else;
      if strcmp(dist_str,'beta')==0;
         fprintf(fid,[prt_str_2 '    ' prt_val ')\n'],dist_add,dist_str,prior_val(1,1),prior_val(1,2),prior_val(1,3));
      else;
         fprintf(fid,[prt_str_2 '    ' prt_val '\n'],dist_add,dist_str,prior_val(1,1),prior_val(1,2),prior_val(1,3));
         fprintf(fid,['                                                        ' prt_val ')\n'],prior_val(1,4));
      end;
   end;
end;
if max(isnan(phiFittedStd))==0;
   AddNote = '      partial derivatives. The second standard deviation is based on fitting a\n      quadratic to the evaluated log posterior and using partial derivatives.\n\n';
else;
   AddNote = '      partial derivatives.\n\n';
end;

fprintf(fid,'============================================================================\n\n');
fprintf(fid,'NOTE: The standard deviation is computed from the inverse Hessian at the posterior\n');
fprintf(fid,['      mode. It has been computed from the inverse Hessian of phi at the mode using\n' AddNote]);
%
% Print the maximum eigenvalues at the mode and at the initial values
%
fprintf(fid,'==============================================================\n');
fprintf(fid,' Maximum absolute eigenvalue of the state transition matrix F \n');
fprintf(fid,'--------------------------------------------------------------\n');
fprintf(fid,[' At initial parameter values:   ' prt_val '\n'],MaxEigenvalueIV);
fprintf(fid,[' At posterior mode values:      ' prt_val '\n'],MaxEigenvalueMode);
fprintf(fid,'==============================================================\n\n');
%
% Now we turn to ExtremeCorrParameters, a cell array containing the parameters with correlations above or equal to
% 0.7 at the mode 
%
if isempty(ExtremeCorrParameters)==0;
   prt_str = ['      %s%s    %s%s    ' prt_val '\n'];
   fprintf(fid,'=======================================================================\n');
   fprintf(fid,' Parameters with an absolute correlation at the mode of 0.7 or greater\n\n');
   fprintf(fid,'              parameter 1             parameter 2      correlation\n');
   fprintf(fid,'-----------------------------------------------------------------------\n');
   for i=1:size(ExtremeCorrParameters,1);
      fprintf(fid,prt_str,add_str,ExtremeCorrParameters{i,1},add_str,ExtremeCorrParameters{i,2},ExtremeCorrParameters{i,3});
   end;
   fprintf(fid,'=======================================================================\n\n');
end;
%
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*                        F O R E C A S T   E R R O R S                         *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% print results about the mean squared errors
%
prt_str = ['%s%s   ' prt_val '   ' prt_val '   ' prt_val '   ' prt_val '\n'];
if size(DSGEModel.VariableNames,2)<15;
   add_str = SpaceStr(15-size(DSGEModel.VariableNames,2));
else;
   add_str = '';
end;
fprintf(fid,'=============================================================================\n');
fprintf(fid,'                      Mean Forecast Error       Mean Squared Forecast Error  \n');
fprintf(fid,'    Variable          mode          init val        mode          init val   \n');
fprintf(fid,'-----------------------------------------------------------------------------\n');
for i=1:size(DSGEModel.VariableNames,1);
   fprintf(fid,prt_str,add_str,DSGEModel.VariableNames(i,:),MFEYMode(i,1),MFEYIV(i,1),MSEYMode(i,i),MSEYIV(i,i));
end;
fprintf(fid,'=============================================================================\n\n');
%
% Some information about the filename and its location
%
fprintf(fid,'Directory for file: %s\n',GetPath(ModeTextFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(ModeTextFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintPosteriorModeSummary.m
%
