function PrintParameterCovarianceMatrix(TxtFile,ParameterNames,ParameterCovarianceMatrix,InverseHessian,NumBurnin,NumDraws,CurrINI)
% PrintParameterCovarianceMatrix: Writes the standard deviations and correlations for the estimated parameter
%                                 covariance matrix to a text-file. As a comparison, the standard deviations and
%                                 correlations based on the inverse Hessian at the posterior mode are also
%                                 presented.
%
% USAGE:
%
%       PrintParameterCovarianceMatrix(TxtFile,ParameterNames,ParameterCovarianceMatrix,InverseHessian,NumBurnin,NumDraws,CurrINI)
%
% REQUIRED INPUT: TxtFile (string vector) with the full path and name of the text-file to write the results
%                         to.
%
%                 ParameterNames (string matrix) of dimension pxp1 with the names of the parameters.
%
%                 ParameterCovarianceMatrix (matrix) of dimension pxp with the estimated covariances based
%                         on the post-burnin-period posterior draws of the transformed parameters.
%
%                 InverseHessian (matrix) of dimension pxp with the estimated inverse Hessian for the
%                         transformed parameters at the posterior mode.
%
%                 NumBurning (integer) with the number of burnin draws.
%
%                 NumDraws (integer) with the total number of posterior draws.
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: February 12, 2009.
%                        This version: February 13, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-02-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% create the file
%
fid = fopen(TxtFile,'wt');
%
% print the header
%
fprintf(fid,'*********************************************************************************************************\n');
fprintf(fid,'*                                                                                                       *\n');
fprintf(fid,'*                         P A R A M E T E R   C O V A R I A N C E   M A T R I X                         *\n');
fprintf(fid,'*                                                                                                       *\n');
fprintf(fid,'*********************************************************************************************************\n\n');
%
fprintf(fid,'Number of burnin draws: %0.0f\n',NumBurnin);
fprintf(fid,'Number of used draws:   %0.0f\n\n',NumDraws-NumBurnin);
%
% Compute the standard deviations
%
PStdVec = sqrt(diag(ParameterCovarianceMatrix));
IHStdVec = sqrt(diag(InverseHessian));
%
% now compute the correlations
%
PCorrMat = ParameterCovarianceMatrix./(PStdVec*PStdVec');
IHCorrMat = InverseHessian./(IHStdVec*IHStdVec');
%
% now write the staandard deviations to file
%
if size(ParameterNames,2)<20;
   AddSpace = SpaceStr(20-size(ParameterNames,2));
else;
   AddSpace = '';
end;
PrtVal = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
PrtStr = ['%s%s     ' PrtVal '       ' PrtVal '\n'];
%
TopStr = MultiCharStr('=',61);
SepStr = MultiCharStr('-',61);
%
% print the results
%
fprintf(fid,'%s\n',TopStr);
fprintf(fid,'                                 Standard deviation \n');
fprintf(fid,'        Parameter        Sample estimate    Inverse Hessian\n');

for i=1:length(PStdVec);
   fprintf(fid,'%s\n',SepStr);
   fprintf(fid,PrtStr,AddSpace,ParameterNames(i,:),PStdVec(i),IHStdVec(i));
end;
fprintf(fid,'%s\n\n',TopStr);
%
% check which parameters have an absolute correlation greater than or equal 0.7
%
ExtremeCorrBound = 0.5;
CorrMatrix = tril(PCorrMat,-1);
IHCorrMatrix = tril(IHCorrMat,-1);
[RowIndex,ColIndex] = find(abs(CorrMatrix)>=ExtremeCorrBound);
if isempty(RowIndex)==0;
   ExtremeCorrParameters = cell(length(RowIndex),3);
   for i=1:length(RowIndex)
      ExtremeCorrParameters{i,1} = ParameterNames(RowIndex(i),:);
      ExtremeCorrParameters{i,2} = ParameterNames(ColIndex(i),:);
      ExtremeCorrParameters{i,3} = CorrMatrix(RowIndex(i),ColIndex(i));
      ExtremeCorrParameters{i,4} = IHCorrMatrix(RowIndex(i),ColIndex(i));
   end;
else;
   ExtremeCorrParameters = [];
end;
%
% print correlations that are greater than 0.5
%
if isempty(ExtremeCorrParameters)==0;
   TopStr = MultiCharStr('=',70);
   SepStr = MultiCharStr('-',70);
   %
   PrtStr = ['      %s%s    %s%s      %' num2str(3+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f\n'];
   PrtStr2 = ['                                                       (%' num2str(3+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f)\n'];
   %
   fprintf(fid,'%s\n',TopStr);
   fprintf(fid,' Parameters with an absolute correlation of at least 0.50 or greater \n');
   fprintf(fid,'              parameter 1             parameter 2      correlation\n');
   for i=1:size(ExtremeCorrParameters,1);
      fprintf(fid,'%s\n',SepStr);
      fprintf(fid,PrtStr,AddSpace,ExtremeCorrParameters{i,1},AddSpace,ExtremeCorrParameters{i,2},ExtremeCorrParameters{i,3});
      fprintf(fid,PrtStr2,ExtremeCorrParameters{i,4});
   end;
   fprintf(fid,'%s\n\n',TopStr);
   fprintf(fid,'NOTE: The numbers within parenthesis are taken from the inverse Hessian\n      at the posterior mode.\n\n');
end;
%
% Some information about the filename and its location
%
fprintf(fid,'Directory for file: %s\n',GetPath(TxtFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(TxtFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);


%
% end of PrintParameterCovarianceMatrix.m
%
