function PrintExtremeCorrelations(TextFile,HeaderStr,ExtremeCorrParameters,ExtremeCorrBound,MatrixRank,NumParam,CurrINI)
% PrintExtremeCorrelations: Write a table with large correlations to a text file.
%
% USAGE:
%
%       PrintExtremeCorrelations(TextFile,HeaderStr,ExtremeCorrParameters,ExtremeCorrBound,MatrixRank,NumParam,CurrINI)
%
% REQUIRED INPUT: TextFile (string vector) with the full path and name of a text file that output
%                          is written to.
%
%                 HeaderStr (string matrix) with the text to use in the header of the table. Each
%                          printed line should have length 80 for best output result and should start
%                          and end with a "*" character.
%
%                 ExtremeCorrParameters (cell array) with parameter pairs having absolute correlations
%                          equal to or greater than "ExtremeCorrBound". The dimension of the cell array
%                          is equal to the number of such extreme correlations times 3. The columns
%                          contain the name of parameter or variable 1 (string), the name of parameter
%                          or variable 2 (string), and the correlation (scalar).
%
%                 ExtremeCorrBound (string vector) with value between 0 and 1 giving the lower bound
%                          for a correlation to be viewed as high.
%
%                 MatrixRank (integer) with the rank of the underlying covariance matrix.
%
%                 NumParam (integer) with the number of rows and columns of the underlying covariance
%                          matrix.
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: February 13, 2008.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% open the file
%
fid = fopen(TextFile,'wt');
%
% write the header of the file
%
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
for i=1:size(HeaderStr,1);
   fprintf(fid,'%s\n',HeaderStr(i,:));
end;
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
ns = size(ExtremeCorrParameters{1,1},2);
if ns<20;
   add_str = SpaceStr(20-ns);
else;
   add_str = '';
end;
prt_val = ['%' num2str(9+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
prt_str = ['      %s%s    %s%s    ' prt_val '\n'];
fprintf(fid,'=======================================================================\n');
fprintf(fid,' Parameters with an absolute correlation of at least %s or greater\n\n',ExtremeCorrBound);
fprintf(fid,'              parameter 1             parameter 2      correlation\n');
fprintf(fid,'-----------------------------------------------------------------------\n');
for i=1:size(ExtremeCorrParameters,1);
   fprintf(fid,prt_str,add_str,ExtremeCorrParameters{i,1},add_str,ExtremeCorrParameters{i,2},ExtremeCorrParameters{i,3});
end;
fprintf(fid,'=======================================================================\n\n');
if MatrixRank<NumParam;
   %
   % write the rank reduction issue
   %
   fprintf(fid,'NOTE: The underlying covariance matrix is estimated to have rank %0.0f\n',MatrixRank);
   fprintf(fid,'      while a full rank matrix should have rank %0.0f. The inverse can\n',NumParam);
   fprintf(fid,'      still be calculated since the rank estimate is based on a\n');
   fprintf(fid,'      tolerance level smaller than 1.0e-010.\n\n');
   fprintf(fid,'=======================================================================\n\n');
end;
%
% Some information about the filename and its location
%
fprintf(fid,'Directory for file: %s\n',GetPath(TextFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(TextFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintExtremeCorrelations.m
%
