function PrintDSGEModestyStatistics(DSGEModel,CurrINI,CurrChain,MaxPH,NumPeriods,NumDraws,NumPredPaths,MultiPValue,MeanMultiModesty,MeanUniModesty,MeanUniModestyLZ,ShockMean,ShockNames)
% PrintDSGEModestyStatistics: Prints mutlivariate and univariate modesty results to text file.
%
% USAGE:
%
%       PrintDSGEModestyStatistics(DSGEModel,CurrINI,CurrChain,MaxPH,NumPeriods,NumDraws,NumPredPaths,MultiPValue,MeanMultiModesty,MeanUniModesty,MeanUniModestyLZ,ShockMean,ShockNames)
%
% REQUIRED INPUT: DSGEModel (structure)  whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 CurrINI (structure) with initialization information.
%
%                 CurrChain (integer) with the number for the MCMC chain used. It can also be a string taking
%                           the values "Posterior Mode" or "Initial Values"
%
%                 MaxPH (integer) with the maximum prediction horizon.
%
%                 NumPeriods (integer) with the number of conditional predictions.
%
%                 NumDraws (integer) with the total number of parameter draws.
%
%                 NumPredPaths (integer) with tht total number of prediction paths per parameter draw.
%
%                 MultiPValue (scalar) with the probability that the multivariate modesty statistic
%                           is smaller than one based on unconditional shocks.
%
%                 MeanMultiModesty (vector) with 2 elements. The first is the mean of the multivariate
%                           modesty statistic based on the conditioning consistent shocks and the second
%                           is the mean of this statistic for randomly drawn shocks.
%
%                 MeanUniModesty (vector) with means of the univariate modesty statistics for all observed
%                           variables.
%
%                 MeanUniModestyLZ (vector) with the means of the univariate modesty statistics suggested
%                           by Leeper-Zha (other shocks are zero over examined prediction horizon) for
%                           all observed variables.
%
%                 ShockMean (matrix) of dimension q x MaxPH with the mean values of the shocks under conditioning.
%
%                 ShockNames (string matrix) of dimension q x qstr with the names of the economic shocks.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: March 12, 2007.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-3-2007:  Finished the function.
%
% * 27-3-2007:  The function can now print results for the modesty statistics based on given parameter values.
%
% * 13-11-2007: Updated the documentation.
%
% * 17-12-2007: The filename now taken the shock control method into account.
%
% * 18-12-2007: Added the "ShockMean" input and code for printing the mean shock values under conditioning.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-7-2008:  Added check for the existence of relevant chi-square distribution functions.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% now we should write the results to a text file.
%
if DSGEModel.ShockControlMethod==1;
   ShockMethod = 'Direct-';
else;
   ShockMethod = 'WaggonerZha-';
end;
if isnumeric(CurrChain)==1;
   file = [DSGEModel.OutputDirectory '\DSGE-ModestyStatistics-' ShockMethod DSGEModel.NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.txt'];;
else;
   file = [DSGEModel.OutputDirectory '\DSGE-ModestyStatistics-' ShockMethod DSGEModel.NameOfModel '-' num2str(MaxPH,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '-' strrep(CurrChain,' ','') '.txt'];;
end;
fid = fopen(file,'wt');
%
% the header information
%
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*                     M O D E S T Y   S T A T I S T I C S                      *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% Model info
%
fprintf(fid,'Name of model:                    %s\n',DSGEModel.NameOfModel);
fprintf(fid,'Data construction file:           %s\n',GetFilename(DSGEModel.DataConstructionFile));
fprintf(fid,'Measurement equation file:        %s\n',GetFilename(DSGEModel.MeasurementEquationFile));
fprintf(fid,'Prior distribution data file:     %s\n',GetFilename(DSGEModel.PriorFile));
fprintf(fid,'Sheet name:                       %s\n',DSGEModel.PriorFileSheet);
if FileExist(DSGEModel.UpdateParameterFile)==1;
   fprintf(fid,'Update parameters file:           %s\n',GetFilename(DSGEModel.UpdateParameterFile));
end;
if FileExist(DSGEModel.InitializeParameterFile)==1;
   fprintf(fid,'Initialize parameters file:       %s\n',GetFilename(DSGEModel.InitializeParameterFile));
end;
fprintf(fid,'AiM model file:                   %s\n\n',GetFilename(DSGEModel.AIMFile));
if NumDraws>1;
   fprintf(fid,'Number of used parameter draws:   %0.0f\n',NumDraws);
   fprintf(fid,'Prediction paths/parameter draw:  %0.0f\n',NumPredPaths);
   fprintf(fid,'Total number of prediction paths: %0.0f\n\n',NumDraws*NumPredPaths);
else;
   fprintf(fid,'Total number of prediction paths: %0.0f\n\n',NumPredPaths);
end;
%
% the multivariate modesty statistics and tail probability
%
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
fprintf(fid,'================================================================================\n');
fprintf(fid,['Tail probability for multivariate modesty statistic: ' prt_val '\n'],MultiPValue);
fprintf(fid,'--------------------------------------------------------------------------------\n');
fprintf(fid,['Mean of multivariate modesty statistic:              ' prt_val '\n'],MeanMultiModesty(1));
fprintf(fid,['Mean of multivariate reference statistic:            ' prt_val '\n'],MeanMultiModesty(2));
fprintf(fid,'--------------------------------------------------------------------------------\n');
if exist('YADAchi2cdf')==2;
   fprintf(fid,['chi-square(%2.0f) tail probability at mean:             ' prt_val '\n'],DSGEModel.n,1-YADAchi2cdf(MeanMultiModesty(1),DSGEModel.n));
else;
   if exist('chi2cdf')==2;
      fprintf(fid,['chi-square(%2.0f) tail probability at mean:             ' prt_val '\n'],DSGEModel.n,1-chi2cdf(MeanMultiModesty(1),DSGEModel.n));
   else;
      fprintf(fid,['chi-square(%2.0f) tail probability at mean:             ' prt_val '\n'],DSGEModel.n,1-YADACheapchi2cdf(MeanMultiModesty(1),DSGEModel.n));
   end;
end;
fprintf(fid,'================================================================================\n\n');
%
% the univariate statistics
%
if size(DSGEModel.VariableNames,2)<15;
   add_name_str = SpaceStr(15-size(DSGEModel.VariableNames,2));
else;
   add_name_str = '';
end;
fprintf(fid,'===================================================\n');
fprintf(fid,'           Univariate Modesty Statistics\n');
fprintf(fid,' Variable            ALLV-statistic   LZ-statistic\n');
fprintf(fid,'---------------------------------------------------\n');
for i=1:length(MeanUniModesty);
   fprintf(fid,[' %s%s     ' prt_val '     ' prt_val '\n'],DSGEModel.VariableNames(i,:),add_name_str,MeanUniModesty(i),MeanUniModestyLZ(i));
end;
fprintf(fid,'===================================================\n\n');
if DSGEModel.ShockControlMethod==1;
   fprintf(fid,'NOTE: The ALLV-statistic is the univariate modesty\n      statistic suggested by Adolfson, Lasen, Lind\n      and Villani (2005) in International Finance.\n      The LZ-statistic is the Leeper-Zha modesty\n      statistic. The former statistic is based on\n      drawing shock values for the unrestricted    \n      economic shocks and the measurement errors,   \n      while the latter sets these shocks to zero.  \n      The reference distribution for these statistics\n      is the standard normal.\n\n');
else;
   fprintf(fid,'NOTE: The ALLV-statistic is the univariate modesty\n      statistic suggested by Adolfson, Lasen, Lind\n      and Villani (2005) in International Finance,\n      and modified by Warne (2007) to handle the\n      Waggoner-Zha shock distribution. The LZ-   \n      statistic is the Leeper-Zha modesty statistic\n      using the mean shock values from the Waggoner-\n      Zha shock distribution under zero measurement\n      errors and mean initial state vector for the   \n      predictions. The reference distribution\n      for these statistics is the standard normal.\n\n');
end;
%
% check if we have data for printing the mean shock values
%
if isempty(ShockMean)==0;
   q = size(ShockMean,1);
   if size(ShockNames,2)<20;
      AddNameStr = SpaceStr(20-size(ShockNames,2));
   else;
      AddNameStr = '';
   end;
   HeadStr = '               Shock';
   PrtStr = '%s%s';
   for i=1:MaxPH;
      if (IsInteger(i/6)==0)|(i==MaxPH);
         HeadStr = [HeadStr sprintf('        %2.0f     ',i)];
         PrtStr = [PrtStr '   ' prt_val];
      else;
         HeadStr = [HeadStr sprintf('        %2.0f     \n                    ',i)];
         PrtStr = [PrtStr '   ' prt_val '\n                    '];
      end;
   end;
   TopStr = MultiCharStr('=',min(111,length(HeadStr)+1));
   DivStr = MultiCharStr('-',min(111,length(HeadStr)+1));
   fprintf(fid,'%s\n',TopStr);
   if isnumeric(CurrChain)==1;
      fprintf(fid,'                    Mean value of shocks at mean prediction under conditioning\n\n');
   else;
      fprintf(fid,'                    Value of shocks at mean prediction under conditioning\n\n');
   end;
   fprintf(fid,'                            Prediction Horizon\n');
   fprintf(fid,'%s\n',HeadStr);
   fprintf(fid,'%s\n',TopStr);
   for i=1:q;
      fprintf(fid,[PrtStr ' \n'],AddNameStr,ShockNames(i,:),ShockMean(i,:)');
      if i<q;
         fprintf(fid,'%s\n',DivStr);
      else;
         fprintf(fid,'%s\n\n',TopStr);
      end;
   end;
end;
%
% we're all done now
%
fprintf(fid,'Directory for file: %s\n',GetPath(file));
fprintf(fid,'Name of file:       %s\n',GetFilename(file));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintDSGEModestyStatistics.m
%
