function PrintConditionalCVD(DSGEModel,CurrINI,CVDs,PrintGroups,VarType,NumDraws,TotalDraws,CurrChain)
% PrintConditionalCVD: Writes the conditional forecast error variance decompositions for the observed
%                      variables to a text file.
%
% USAGE:
%
%       PrintConditionalCVD(DSGEModel,CurrINI,CVDs,PrintGroups,VarType,NumDraws,TotalDraws,CurrChain)
%
% REQUIRED INPUT: DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 CurrINI (structure), with initialization information.
%
%                 CVDs (structure) with fields FEVD (posterior mean), Quantiles(j).FEVD,
%                           and Quantiles(j).percent with the quantile and percent values
%                           of the posterior distribution.
%
%                 PrintGroups (boolean) which determines if shocks groups should be printed (1) or not (0).
%
%                 VarType (string vector) that takes on the values "Original", "Levels",
%                           or "Annualized".
%
%                 NumDraws (integer) with the number of draws from the posterior distribution
%                           of the DSGE model parameters that were used in the computations.
%
%                 TotalDraws (integer) with the total number of draws from the posterior
%                           distribution of the DSGE model parameters that could have been
%                           used for the computations.
%
%                 CurrChain (integer) with the value of the current MCMC chain.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: April 4, 2007.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-11-2007: Updated the documentation.
%
% * 1-4-2008:   Fixed a spelling error.
%
% * 14-4-2008:  The print code takes into account that the last decomposition for the levels data
%               is an estimate of the long run.
%
% * 17-4-2008:  Added the input variable "PrintGroups".
%
% * 23-4-2008:  Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if PrintGroups==0;
   ResFile = [DSGEModel.OutputDirectory '\PosteriorDist-CVD-' VarType '-' num2str(CurrChain,'%0.0f') '-' DSGEModel.NameOfModel '-' num2str(NumDraws,'%0.0f') '.' num2str(TotalDraws,'%0.0f') '.txt'];
else;
   ResFile = [DSGEModel.OutputDirectory '\PosteriorDist-CVD-ShockGroups-' VarType '-' num2str(CurrChain,'%0.0f') '-' DSGEModel.NameOfModel '-' num2str(NumDraws,'%0.0f') '.' num2str(TotalDraws,'%0.0f') '.txt'];
end;
fid = fopen(ResFile,'wt');
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*    C O N D I T I O N A L   V A R I A N C E   D E C O M P O S I T I O N S     *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
fprintf(fid,'Number of parameter draws used:      %0.0f \n',NumDraws);
fprintf(fid,'Number of available parameter draws: %0.0f \n',TotalDraws);
fprintf(fid,'Current MCMC chain:                  %0.0f \n\n',CurrChain);
%
fprintf(fid,'NOTE: The conditional forecast error variance decomposition neglect the state\n      variable uncertainty and the measurement errors. They may therefore\n      differ greatly from the forecast error variance decompositions.\n\n');
%
fprintf(fid,'      Posterior mean values of conditional variance decompositions for the %s\n      data along with quantiles.\n\n',lower(VarType));
%
[n,q,h] = size(CVDs.FEVD);
vn = size(DSGEModel.VariableNames,2);
if PrintGroups==0;
   vq = size(DSGEModel.StateShockNames,2);
   NamesMatrix = DSGEModel.StateShockNames;
else;
   vq = size(DSGEModel.ShockGroupNames,2);
   NamesMatrix = char(zeros(size(DSGEModel.ShockGroupNames)));
   for i=1:q;
      NamesMatrix(i,:) = strrep(DSGEModel.ShockGroupNames(i,:),'_',' ');
   end;
end;
if vn<15;
   add_name_str = SpaceStr(15-vn);
else;
   add_name_str = '';
end;
if vq<20;
   add_shock_str = SpaceStr(20-vq);
else;
   add_shock_str = '';
end;
NameStr = '                    ';
MEStr = '';
prt_str = '%s';
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
q_prt_str = '       %4.1f quantile';
for i=1:n;
   NameStr = [NameStr '   ' add_name_str DSGEModel.VariableNames(i,:)];
   prt_str = [prt_str '      ' prt_val];
   q_prt_str = [q_prt_str '      ' prt_val];
end;
%
for i=1:q;
   MEStr = strvcat(MEStr,sprintf('%s%s',add_shock_str,NamesMatrix(i,:)));
end;
FirstStr = MultiCharStr('=',length(NameStr)+1);
DivStr = MultiCharStr('-',length(NameStr)+1);
%
% check if we should relate the shocks to shock groups
%
if PrintGroups==1;
   for i=1:q;
      iGroup = (DSGEModel.ShockGroups==i) .* (1:length(DSGEModel.ShockGroups));
      iGroup = iGroup(iGroup>0);
      ShockNames = StrWrap(vecstr(DSGEModel.StateShockNames(iGroup,:),' '),60);
      GroupName = StringTrim(NamesMatrix(i,:));
      if size(GroupName,2)<25;
         AddSpace = SpaceStr(25-size(GroupName,2));
      else;
         AddSpace = '';
      end;
      fprintf(fid,'%s:%s %s\n',GroupName,AddSpace,ShockNames(1,:));
      for j=2:size(ShockNames,1);
         if strcmp(StringTrim(ShockNames(j,:)),'')==0;
            fprintf(fid,'%s%s\n',SpaceStr(27),ShockNames(j,:));
         end;
      end;
   end;
   fprintf(fid,'\n');
end;
%
% Write to output for horizon h
%
for j=1:h;
   fprintf(fid,'%s\n',FirstStr);
   if (strcmp(lower(VarType),'levels')==0)|(j<h);
      fprintf(fid,'Forecast Horizon: %3.0f\n',j);
   else;
      fprintf(fid,'Forecast Horizon: Long run\n');
   end;
   fprintf(fid,'%s\n',NameStr);
   for i=1:q;
      fprintf(fid,'%s\n',DivStr);
      fprintf(fid,[prt_str '\n'],MEStr(i,:),abs(CVDs.FEVD(:,i,j)));
      for k=1:length(CVDs.Quantiles);
         fprintf(fid,[q_prt_str '\n'],CVDs.Quantiles(k).percent,abs(CVDs.Quantiles(k).FEVD(:,i,j)));
      end;
   end;
   fprintf(fid,'%s\n',FirstStr);
   fprintf(fid,[prt_str '\n'],'All Economic Shocks ',sum(abs(CVDs.FEVD(:,:,j)),2));
   fprintf(fid,'%s\n\n',FirstStr);
end;
%
% we're all done now
%
fprintf(fid,'Directory for file: %s\n',GetPath(ResFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(ResFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintConditionalCVD.m
%
