function PrintCondVarianceDecompositions(DSGEModel,FEVDs,ConvValue,PrintGroups,CurrINI,EstStr,LevStr)
% PrintCondVarianceDecompositions: Writes the conditional forecast error variance decompositions for the observed
%                                  variables to a text file.
%
% USAGE:
%
%       PrintCondVarianceDecompositions(DSGEModel,FEVDs,ConvValue,PrintGroups,CurrINI,EstStr,LevStr)
%
% REQUIRED INPUT: DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 FEVDs (nxqxh) matrix with the forecast error variance decompositions. For each h, the nxq matrix is
%                           ordered such the the q economic are in the columns and the n observed variables in the rows.
%
%                 ConvValue (nx1) vector with the ratio between the conditional forecast error variance at period h
%                           and the estimate of the long-run forecast error variance.
%
%                 PrintGroups (boolean) which determines if shocks groups should be printed (1) or not (0).
%
%                 CurrINI (structure), with initialization information.
%
%                 EstStr (string vector) that takes the value "posterior mode" or "initial".
%
%                 LevStr (string vector) that is either empty, "Annual" or "Levels". The latter means that all
%                            observed variable are viewed in levels.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: December 4, 2006.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 8-12-2006:  Added the LevStr input variable.
%
% * 4-4-2007:   Updated the documentation.
%
% * 13-11-2007: Updated the documentation.
%
% * 14-4-2008:  The print code takes into account that the last decomposition for the levels data
%               is an estimate of the long run. Moreover, the "ConvValue" input was also added.
%
% * 16-4-2008:  The input variable "PrintGroups" is now taken into account.
%
% * 17-4-2008:  Added the code for writing to file the shocks that belong to a each shock group.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if isempty(LevStr)==0;
   LevStr = [LevStr '-'];
end;
if strcmp(lower(EstStr),'posterior mode')==1;
   if PrintGroups==0;
      ResFile = [DSGEModel.OutputDirectory '\PosteriorModeCFEVDs-' LevStr DSGEModel.NameOfModel '.txt'];
   else;
      ResFile = [DSGEModel.OutputDirectory '\PosteriorModeCFEVDs-ShockGroups-' LevStr DSGEModel.NameOfModel '.txt'];
   end;
else;
   if PrintGroups==0;
      ResFile = [DSGEModel.OutputDirectory '\InitialValueCFEVDs-' LevStr DSGEModel.NameOfModel '.txt'];
   else;
      ResFile = [DSGEModel.OutputDirectory '\InitialValueCFEVDs-ShockGroups-' LevStr DSGEModel.NameOfModel '.txt'];
   end;
end;
fid = fopen(ResFile,'wt');
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*    C O N D I T I O N A L   V A R I A N C E   D E C O M P O S I T I O N S     *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
fprintf(fid,'NOTE: The conditional forecast error variance decomposition neglect the state\n      variable uncertainty and the measurement errors. They may therefore\n      differ greatly from the forecast error variance decompositions.\n\n');
%
[n,q,h] = size(FEVDs);
vn = size(DSGEModel.VariableNames,2);
if PrintGroups==0;
   vq = size(DSGEModel.StateShockNames,2);
   NamesMatrix = DSGEModel.StateShockNames;
else;
   vq = size(DSGEModel.ShockGroupNames,2);
   NamesMatrix = char(zeros(size(DSGEModel.ShockGroupNames)));
   for i=1:q;
      NamesMatrix(i,:) = strrep(DSGEModel.ShockGroupNames(i,:),'_',' ');
   end;
end;
if vn<15;
   add_name_str = SpaceStr(15-vn);
else;
   add_name_str = '';
end;
if vq<20;
   add_shock_str = SpaceStr(20-vq);
else;
   add_shock_str = '';
end;
NameStr = '                    ';
MEStr = '';
prt_str = '%s';
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
for i=1:n;
   NameStr = [NameStr '   ' add_name_str DSGEModel.VariableNames(i,:)];
   prt_str = [prt_str '      ' prt_val];
end;
%
for i=1:q;
   MEStr = strvcat(MEStr,sprintf('%s%s',add_shock_str,NamesMatrix(i,:)));
end;
FirstStr = MultiCharStr('=',length(NameStr));
DivStr = MultiCharStr('-',length(NameStr));
%
% check if we should relate the shocks to shock groups
%
if PrintGroups==1;
   for i=1:q;
      iGroup = (DSGEModel.ShockGroups==i) .* (1:length(DSGEModel.ShockGroups));
      iGroup = iGroup(iGroup>0);
      ShockNames = StrWrap(vecstr(DSGEModel.StateShockNames(iGroup,:),' '),60);
      GroupName = StringTrim(NamesMatrix(i,:));
      if size(GroupName,2)<25;
         AddSpace = SpaceStr(25-size(GroupName,2));
      else;
         AddSpace = '';
      end;
      fprintf(fid,'%s:%s %s\n',GroupName,AddSpace,ShockNames(1,:));
      for j=2:size(ShockNames,1);
         if strcmp(StringTrim(ShockNames(j,:)),'')==0;
            fprintf(fid,'%s%s\n',SpaceStr(27),ShockNames(j,:));
         end;
      end;
   end;
   fprintf(fid,'\n');
end;
%
% Write to output for horizon h
%
for j=1:size(FEVDs,3);
   fprintf(fid,'%s\n',FirstStr);
   if (strcmp(lower(LevStr),'levels-')==0)|(j<size(FEVDs,3));
      fprintf(fid,'Forecast Horizon: %3.0f\n',j);
   else;
      fprintf(fid,'Forecast Horizon: Long run\n');
   end;
   fprintf(fid,'%s\n',NameStr);
   for i=1:q;
      fprintf(fid,'%s\n',DivStr);
      fprintf(fid,[prt_str '\n'],MEStr(i,:),abs(FEVDs(:,i,j)));
   end;
   fprintf(fid,'%s\n',FirstStr);
   fprintf(fid,[prt_str '\n'],'All Economic Shocks ',sum(abs(FEVDs(:,:,j)),2));
   if (strcmp(lower(LevStr),'levels-')==0)|(j~=size(FEVDs,3)-1);
      fprintf(fid,'%s\n\n',FirstStr);
   else;
      fprintf(fid,'%s\n',FirstStr);
      fprintf(fid,[prt_str '\n'],'Long-Run convergence',ConvValue');
      fprintf(fid,'%s\n\n',FirstStr);
   end;
end;
%
if isempty(ConvValue)==0;
   fprintf(fid,'NOTE: Long-run convergence measures the %0.0f-steps ahead conditional forecast error variance\n      relative to the estimated long-run conditional forecast error variance. For\n      first differenced variables the levels forecast error variances are divided by\n      the forecast horizon. Values close to unity indicates convergence, while very\n      large or very small values are at odds with convergence to the long-run.\n\n',size(FEVDs,3)-1);
end;
%
% we're all done now
%
fprintf(fid,'Directory for file: %s\n',GetPath(ResFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(ResFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintCondVarianceDecompositions.m
%
