function PrintCondCorrelations(DSGEModel,CorrData,CurrINI,EstStr,NumPaths,NumParam)
% PrintCondCorrelations: Writes the results from calculating conditional correlations to disk.
%
% USAGE:
%
%       PrintCondCorrelations(DSGEModel,CorrData,CurrINI,EstStr,NumPaths,NumParam)
%
% REQUIRED INPUT: DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 CorrData (vector structure) with fields "ShockNames", "Mean" and "Quantiles".
%                           The length of the vector structure (struct array) is equal to
%                           the number of nonzero state shocks.
%
%                 CurrINI (structure), with initialization information.
%
%                 EstStr (string vector) that takes the value "posterior mode", "distribution",
%                           or "initial".
%
%                 NumPaths (integer) with the number of paths used for simulation.
%
%                 NumParam (integer) with the number of parameter draws used for the distribution.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: January 12, 2009.
%                        This version: February 19, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 16-02-2009: Updated the documentation.
%
% * 19-02-2009: Added the input variable "NumParam" to allow for printing results
%               using the posterior distirbution of the parameters.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

AddFileName = '';
if NumPaths>0;
   AddFileName = ['-' int2str(NumPaths)];
end;
if strcmp(lower(EstStr),'posterior mode')==1;
   ResFile = [DSGEModel.OutputDirectory '\PosteriorModeCondCorr-' DSGEModel.NameOfModel AddFileName '.txt'];
elseif strcmp(lower(EstStr),'distribution')==1;
   ResFile = [DSGEModel.OutputDirectory '\PosteriorDistCondCorr-' DSGEModel.NameOfModel AddFileName '.' int2str(NumParam) '.txt'];
else;
   ResFile = [DSGEModel.OutputDirectory '\InitialValueCondCorr-' DSGEModel.NameOfModel AddFileName '.txt'];
end;
fid = fopen(ResFile,'wt');
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
if NumPaths==0;
   fprintf(fid,'*                       P O P U L A T I O N - B A S E D                        *\n');
else;
   fprintf(fid,'*                           S A M P L E - B A S E D                            *\n');
end;
fprintf(fid,'*               C O N D I T I O N A L   C O R R E L A T I O N S                *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% initializing the formatting of output numbers. we usually apply the 12.6f format.
%
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
%
[n,vn] = size(DSGEModel.VariableNames);
if vn<15;
   add_name_str = SpaceStr(15-vn);
else;
   add_name_str = '';
end;
%
if (NumPaths==0)&(NumParam==1);
   fprintf(fid,'NOTE: Using the %s values.\n\n',lower(EstStr));
   %
   prt_str = ['%s%s,%s%s   ' prt_val ' \n'];
   TopStr = MultiCharStr('=',35+(2*CurrINI.decimals));
   DivStr = MultiCharStr('-',35+(2*CurrINI.decimals));
   %
   for s=1:length(CorrData);
      fprintf(fid,'%s\n',TopStr);
      fprintf(fid,'       Conditional Correlations based on\n');
      fprintf(fid,'       nonzero shock: %s\n',StringTrim(CorrData(1).ShockNames(s,:)));
      for i=1:DSGEModel.n-1;
         for j=i+1:DSGEModel.n;
            fprintf(fid,'%s\n',DivStr);
            fprintf(fid,prt_str,add_name_str,DSGEModel.VariableNames(i,:),DSGEModel.VariableNames(j,:),add_name_str,CorrData(s).Mean(i,j));
         end;
      end;
      fprintf(fid,'%s\n\n',TopStr);
   end;
else;
   %
   if NumPaths>0;
      fprintf(fid,'Number of paths for the observed variables:  %0.0f\n',NumPaths);
   end;
   if NumParam==1;
      fprintf(fid,'\nNOTE: Using the %s values.\n\n',lower(EstStr));
   else;
      fprintf(fid,'Number of parameter draws:                   %0.0f\n\n',NumParam);
   end;
   %
   prt_str = ['%s%s,%s%s           ' prt_val ' \n'];
   prt_str_band = ['%3.0f percent                       [' prt_val ',' prt_val '] \n'];
   TopStr = MultiCharStr('=',38+(4*CurrINI.decimals));
   DivStr = MultiCharStr('-',38+(4*CurrINI.decimals));
   %
   NumBounds = floor(length(CorrData(1).Quantiles)/2);
   %
   for s=1:length(CorrData);
      fprintf(fid,'%s\n',TopStr);
      fprintf(fid,'       Conditional Correlations based on\n');
      fprintf(fid,'       nonzero shock: %s\n',StringTrim(CorrData(1).ShockNames(s,:)));
      for i=1:DSGEModel.n-1;
         for j=i+1:DSGEModel.n;
            fprintf(fid,'%s\n',DivStr);
            fprintf(fid,prt_str,add_name_str,DSGEModel.VariableNames(i,:),DSGEModel.VariableNames(j,:),add_name_str,CorrData(s).Mean(i,j));
            for nb=1:NumBounds;
               fprintf(fid,prt_str_band,CorrData(s).Quantiles(NumBounds+nb).percent-CorrData(s).Quantiles(NumBounds-nb+1).percent,CorrData(s).Quantiles(NumBounds-nb+1).Mean(i,j),CorrData(s).Quantiles(NumBounds+nb).Mean(i,j));
            end;
         end;
      end;
      fprintf(fid,'%s\n\n',TopStr);
   end;
end;
%
% bottom file data
%
fprintf(fid,'Directory for file: %s\n',GetPath(ResFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(ResFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintCondCorrelations.m
%
