function PrintAutoCorrelations(DSGEModel,CorrData,CurrINI,EstStr,NumPaths)
% PrintAutoCorrelations: Writes the results from calculating autocorrelations based on the state-space
%                        model and based on the data to disk.
%
% USAGE:
%
%       PrintAutoCorrelations(DSGEModel,CorrData,CurrINI,EstStr,NumPaths)
%
% REQUIRED INPUT: DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 CorrData (structure) with main fields "StateSpace" and "Data". These fields have sub-fields
%                           "Mean", "AutoCov", and "AutoCorr". The main field "StateSpace" refers to the
%                           results based on the state-space model, while the main field "Data" refers to
%                           the results based on the data. Notice that the autocovariance and autocorrelation
%                           results using the data are calculated by de-meaning using the mean from the
%                           state-space model. This should make the results across the two approaches
%                           easier to compare.
%
%                 CurrINI (structure), with initialization information.
%
%                 EstStr (string vector) that takes the value "posterior mode" or "initial".
%
%                 NumPaths (integer) with the number of paths used for simulation.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: October 26, 2006.
%                        This version: January 12, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 6-11-2006:  Corrected a typo.
%
% * 7-11-2006:  Standard deviations are now written to output rather than the covariance matrix.
%
% * 1-8-2007:   Fixed a print bug when printing the mean and the number of deterministic variables
%               exceeds 1.
%
% * 13-11-2007: Updated the documentation.
%
% * 29-2-2008:  The function no longer prints the autocorrelations. The name of the output file
%               depends on if simulated data are used or not.
%
% * 3-3-2008:   Added the code for printing the when NumPaths is positive.
%
% * 23-5-2008:  Updated the documentation.
%
% * 12-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

AddFileName = '';
if NumPaths>0;
   AddFileName = ['-' int2str(NumPaths)];
end;
if strcmp(lower(EstStr),'posterior mode')==1;
   ResFile = [DSGEModel.OutputDirectory '\PosteriorModeAutoCorr-' DSGEModel.NameOfModel AddFileName '.txt'];
else;
   ResFile = [DSGEModel.OutputDirectory '\InitialValueAutoCorr-' DSGEModel.NameOfModel AddFileName '.txt'];
end;
fid = fopen(ResFile,'wt');
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
if NumPaths==0;
   fprintf(fid,'*                       P O P U L A T I O N - B A S E D                        *\n');
else;
   fprintf(fid,'*                           S A M P L E - B A S E D                            *\n');
end;
fprintf(fid,'*            M E A N S   &   S T A N D A R D   D E V I A T I O N S             *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% initializing the formatting of output numbers. we usually apply the 12.6f format.
%
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
%
[n,vn] = size(DSGEModel.VariableNames);
if vn<15;
   add_name_str = SpaceStr(15-vn);
else;
   add_name_str = '';
end;
%
if NumPaths==0;
   fprintf(fid,'NOTE: Using the %s values.\n\n',lower(EstStr));
   %
   % check first if there are exogenous variables
   %
   prt_str = '';
   for i=1:size(CorrData.StateSpace.Mean,2);
      prt_str = [prt_str '   ' prt_val];
   end;
   prt_str = ['%s%s' prt_str '   |' prt_str '\n'];
   TopStr = MultiCharStr('=',19+(30*size(CorrData.StateSpace.Mean,2)));
   DivStr = MultiCharStr('-',19+(30*size(CorrData.StateSpace.Mean,2)));
   %
   fprintf(fid,'%s\n',TopStr);
   fprintf(fid,'%sAverages\n',SpaceStr(15+(15*size(CorrData.StateSpace.Mean,2))));
   fprintf(fid,'%sDSGE Model   |   Data\n',SpaceStr(5+(15*size(CorrData.StateSpace.Mean,2))));
   fprintf(fid,'%s\n',DivStr);
   for i=1:n;
      fprintf(fid,prt_str,add_name_str,DSGEModel.VariableNames(i,:),[CorrData.StateSpace.Mean(i,:) CorrData.Data.Mean(i,:)]');
   end;
   fprintf(fid,'%s\n\n',TopStr);
   if DSGEModel.k>0;
      XVarNames = StrWrap(vecstr(DSGEModel.XVariableNames,' '),40);
   else;
      XVarNames = 'None';
   end;
   fprintf(fid,'Names of exogenous variables:     %s\n',XVarNames(1,:));
   for i=2:size(XVarNames,1);
      if strcmp(StringTrim(XVarNames(i,:)),'')==0;
         fprintf(fid,'                                  %s\n',XVarNames(i,:));
      end;
   end;
   fprintf(fid,'\nNOTE: The data standard deviations and autocorrelations below are based on\n      first correcting for the data mean.\n\n');
   %
   % Now we deal with the standard deviations
   %
   prt_str = ['%s%s   ' prt_val '   |   ' prt_val '\n'];
   TopStr = MultiCharStr('=',49);
   DivStr = MultiCharStr('-',49);
   fprintf(fid,'%s\n',TopStr);
   fprintf(fid,'%sStandard Deviations\n',SpaceStr(25));                         
   fprintf(fid,'%sDSGE Model   |   Data\n',SpaceStr(20));
   fprintf(fid,'%s\n',DivStr);
   for i=1:n;
      fprintf(fid,prt_str,add_name_str,DSGEModel.VariableNames(i,:),sqrt(CorrData.StateSpace.Cov(i,i)),sqrt(CorrData.Data.Cov(i,i)));
   end;
   fprintf(fid,'%s\n\n',TopStr);
else;
   %
   fprintf(fid,'Number of paths for the observed variables:  %0.0f\n\n',NumPaths);
   fprintf(fid,'NOTE: Using the %s values.\n\n',lower(EstStr));
   %
   [n,vn] = size(DSGEModel.VariableNames);
   if vn<20;
      AddNameStr = SpaceStr(20-vn);
   else;
      AddNameStr = '';
   end;
   %
   % deal with the means
   %
   NumBounds = floor(length(CorrData.Quantiles)/2);
   QuantVec = zeros(NumBounds,1);
   QuantStd = zeros(NumBounds,2,n);
   if NumPaths==0;
      if min(size(DSGEModel.X))>0;
         k = size(DSGEModel.X,1);
         for j=1:k;
            Quant(j).Mean = zeros(NumBounds,2,n);
         end;
      else;
         k = 0;
      end;
   else;
      k = size(CorrData.Mean,2);
      for j=1:k;
         Quant(j).Mean = zeros(NumBounds,2,n);
      end;
   end;
   for nb=1:NumBounds;
      QuantVec(NumBounds-nb+1,1) = CorrData.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-CorrData.Quantiles(nb).percent;
      for i=1:n;
         QuantStd(NumBounds-nb+1,1,i) = CorrData.Quantiles(nb).Std(i,1);
         QuantStd(NumBounds-nb+1,2,i) = CorrData.Quantiles(length(DSGEModel.Percentiles)-nb+1).Std(i,1);
         for j=1:k;
            Quant(j).Mean(NumBounds-nb+1,1,i) = CorrData.Quantiles(nb).Mean(i,j);
            Quant(j).Mean(NumBounds-nb+1,2,i) = CorrData.Quantiles(length(DSGEModel.Percentiles)-nb+1).Mean(i,j);
         end;
      end;
   end;
   %
   % define some print string
   %
   prt_std_str = ['%s%s           ' prt_val '          |   ' prt_val ' \n'];
   prt_quant_str = ['%3.0f percent            [' prt_val ',' prt_val ']   |\n'];
   if k>0;
      kn = size(DSGEModel.XVariableNames,2);
      if kn<20;
         AddXNameStr = SpaceStr(20-kn);
      else;
         AddXNameStr = '';
      end;
      %
      % print the mean results
      %
      for j=1:k;
         fprintf(fid,'======================================================================\n');
         fprintf(fid,'                                                  Averages \n');
         fprintf(fid,'                                        Variable: %s%s \n',DSGEModel.XVariableNames(j,:),AddXNameStr);
         fprintf(fid,'Variable                         DSGE Model          |       Data \n');
         fprintf(fid,'======================================================================\n');
         for i=1:n;
            fprintf(fid,prt_std_str,AddNameStr,DSGEModel.VariableNames(i,:),CorrData.Mean(i,j),CorrData.Data.Mean(i,j));
            for nb=1:NumBounds;
               fprintf(fid,prt_quant_str,QuantVec(nb,1),Quant(j).Mean(nb,:,i)');
            end;
            if i<n;
               fprintf(fid,'----------------------------------------------------------------------\n');
            else;
               fprintf(fid,'======================================================================\n\n');
            end;
         end;
      end;
   end;
   %
   % the standard deviations
   %
   fprintf(fid,'======================================================================\n');
   fprintf(fid,'                                             Standard Deviations \n');
   fprintf(fid,'Variable                         DSGE Model          |       Data \n');
   fprintf(fid,'======================================================================\n');
   for i=1:n;
      fprintf(fid,prt_std_str,AddNameStr,DSGEModel.VariableNames(i,:),CorrData.Std(i,1),CorrData.Data.Std(i,1));
      for j=1:NumBounds;
         fprintf(fid,prt_quant_str,QuantVec(j,1),QuantStd(j,:,i)');
      end;
      if i<n;
         fprintf(fid,'----------------------------------------------------------------------\n');
      else;
         fprintf(fid,'======================================================================\n\n');
      end;
   end;
end;
%
% bottom file data
%
fprintf(fid,'Directory for file: %s\n',GetPath(ResFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(ResFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintAutoCorrelations.m
%
