function PrintAnnualData(ResFile,SummaryData,SampleStrMat,AnnualData,YearStrMat,VariableNames,NumDecimals)
% PrintAnnualData: Writes selected annualized data and average annualised data to file
%
% USAGE:
%
%       PrintAnnualData(ResFile,SummaryData,SampleStrMat,AnnualData,YearStrMat,VariableNames,NumDecimals)
%
% REQUIRED INPUT: ResFile (string vector) with the full path plus filename of the text-file that the
%                         results should be written to.
%
%                 SummaryData (matrix) of dimension Txk with the annualized data for T observations and K variables
%
%                 SampleStrMat (string matrix) of dimension Txk1 with the T sample periods.
%
%                 AnnualData (matrix) of dimension Sxk with the S observations on the average annualized data for the
%                         k variables.
%
%                 YearStrMat (string matrix) of dimension Sxk2 with the S sample years.
%
%                 VariableNames (string matrix) of dimension kxk3 with the names of the k variables.
%
%                 NumDecimals (integer) with the number of decimals to use for the output.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: March 31, 2008.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% create the output file
%
fid = fopen(ResFile,'wt');
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*             S U M M A R Y   O F   A N N U A L I Z E D   D A T A              *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% determine some dimensios
%
[T,k] = size(SummaryData);
k1 = size(SampleStrMat,2);
[S,k2] = size(YearStrMat);
%
% define some formatting
%
if size(VariableNames,2)<20;
   SpStr = SpaceStr(20-size(VariableNames,2));
else;
   SpStr = '';
end;
PrtVal = ['%' int2str(NumDecimals+7) '.' int2str(NumDecimals) 'f'];
%
% where data is written
%
DataSpStr = SpaceStr(NumDecimals+7-k1);
DataEqStr = MultiCharStr('=',NumDecimals+7);
%
VarStr = ' Variable           ';
EqStr = MultiCharStr('=',21);
PrtStr = '%s%s';
%
for i=1:T;
   VarStr = [VarStr '   ' DataSpStr SampleStrMat(i,:)];
   EqStr = [EqStr '===' DataEqStr];
   PrtStr = [PrtStr '   ' PrtVal];
end;
MinStr = MultiCharStr('-',length(EqStr));
%
% write the header for the annualized data
%
fprintf(fid,[EqStr '\n']);
fprintf(fid,' Annualized Data \n');
fprintf(fid,[VarStr '\n']);
fprintf(fid,[EqStr '\n']);
for i=1:k;
   fprintf(fid,[PrtStr '\n'],SpStr,VariableNames(i,:),SummaryData(:,i));
   if i<k;
      fprintf(fid,[MinStr '\n']);
   else;
      fprintf(fid,[EqStr '\n\n']);
   end;
end;
%
if isempty(AnnualData)==0;
   DataSpStr = SpaceStr(NumDecimals+7-k2);
   VarStr = ' Variable           ';
   EqStr = MultiCharStr('=',21);
   PrtStr = '%s%s';
   %
   for i=1:S;
      VarStr = [VarStr '   ' DataSpStr YearStrMat(i,:)];
      EqStr = [EqStr '===' DataEqStr];
      PrtStr = [PrtStr '   ' PrtVal];
   end;
   MinStr = MultiCharStr('-',length(EqStr));
   %
   % write the header for the annualized data
   %
   fprintf(fid,[EqStr '\n']);
   fprintf(fid,' Average Annualized Data \n');
   fprintf(fid,[VarStr '\n']);
   fprintf(fid,[EqStr '\n']);
   for i=1:k;
      fprintf(fid,[PrtStr '\n'],SpStr,VariableNames(i,:),AnnualData(:,i));
      if i<k;
         fprintf(fid,[MinStr '\n']);
      else;
         fprintf(fid,[EqStr '\n\n']);
      end;
   end;
end;
%
% bottom file data
%
fprintf(fid,'Directory for file: %s\n',GetPath(ResFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(ResFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintAnnualData.m
%
