function logPrior = logPriorDSGE(theta,thetaDist,PriorDist,LowerBound)
% logPriorDSGE: Calculates the value of the log of the prior density at the value theta
%
% USAGE:
%
%       logPrior = logPriorDSGE(theta,thetaDist,PriorDist,LowerBound)
%
% REQUIRED INPUT:  theta (vector) with all parameters to be estimated
%
%                  thetaDist (vector) with integers 1 if gamma, 2 if beta, 3 if normal, 4 if invgamma, 5 if
%                            truncnormal, and 6 if uniform.
%
%                  PriorDist (structure) with fields "beta", "gamma", "normal", "invgamma", "truncnormal", and "uniform"
%                            where the values are matrices with the parameter values for the distributions. In the case of
%                            the normal, beta, and gamma these parameter values are the mean and the standard deviation.
%                            For the invgamma it is the s and q parameters, where q is a positive integer (degrees of
%                            freedom). Finally, for the left truncated normal there's a third parameter, the lower bound.
%
%                  LowerBound (vector) with lower bound values.
%
% REQUIRED OUTPUT: logPrior (scalar) with the value of the log prior density.
%         
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: August 8, 2006.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 10-8-2006:  Added the optional LowerBound input argument. This is still not actually used.
%
% * 25-8-2006:  Added the call to the log of the uniform density.
%
% * 28-8-2006:  Included the test if logPrior is a real number. If not it is set to NaN.
%
% * 12-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-7-2008:  Added support for the lower bound for the gamma and invgamma distributions. The
%               "LowerBound" variable is now required.
%
% * 22-7-2008:  Made sure that the lower bound is also subtracted from the "s" parameter for the
%               inverse gamma distribution.
%
% * 23-7-2008:  Took into accoun that a general beta prior is now supported.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% find the elements of theta that have gamma, beta, normal, inverse gamma, left truncated normal, and uniform priors
%
gamma_vec = theta(thetaDist==1);
beta_vec = theta(thetaDist==2);
normal_vec = theta(thetaDist==3);
invgamma_vec = theta(thetaDist==4);
truncnormal_vec = theta(thetaDist==5);
uniform_vec = theta(thetaDist==6);
lb_gamma_vec = LowerBound(thetaDist==1);
lb_invgamma_vec = LowerBound(thetaDist==4);
%
logPrior = 0;
if isempty(gamma_vec)==0;
   logPrior = sum(logGammaPDF(gamma_vec-lb_gamma_vec,PriorDist.gamma_ab(:,1),PriorDist.gamma_ab(:,2)));
end;
if isempty(beta_vec)==0;
   if size(PriorDist.beta)==4;
      logPrior = logPrior+sum(logBetaPDF(beta_vec,PriorDist.beta_ab(:,1),PriorDist.beta_ab(:,2),PriorDist.beta(:,3),PriorDist.beta(:,4)));
   else;
      logPrior = logPrior+sum(logBetaPDF(beta_vec,PriorDist.beta_ab(:,1),PriorDist.beta_ab(:,2),zeros(length(beta_vec),1),ones(length(beta_vec),1)));
   end;
end;
if isempty(normal_vec)==0;
   logPrior = logPrior+sum(logNormalPDF(normal_vec,PriorDist.normal(:,1),PriorDist.normal(:,2)));
end;
if isempty(invgamma_vec)==0;
   logPrior = logPrior+sum(logInverseGammaPDF(invgamma_vec-lb_invgamma_vec,PriorDist.invgamma(:,1)-lb_invgamma_vec,PriorDist.invgamma(:,2)));
end;
if isempty(truncnormal_vec)==0;
   logPrior = logPrior+sum(logLTNormalPDF(truncnormal_vec,PriorDist.truncnormal(:,1),PriorDist.truncnormal(:,2),PriorDist.truncnormal(:,3)));
end;
if isempty(uniform_vec)==0;
   logPrior = logPrior+sum(logUniformPDF(PriorDist.uniform(:,1),PriorDist.uniform(:,2)));
end;
%
% check if the log prior density value is real
%
if isreal(logPrior)==0;
   logPrior = NaN;
end;

%
% end of logPriorDSGE.m
%
