function logPost = logPosteriorPhiDSGE(phi,thetaIndex,UniformBounds,LowerBound,thetaPositions,thetaDist,PriorDist,ModelParameters,DSGEModel,AIMData)
% logPosteriorPhiDSGE: computes the value of the log posterior at phi when we take
%                              the log of the Jacobian in the transformation theta = g(phi)
%                              into account.
%
% USAGE:
%
%       logPost = logPosteriorPhiDSGE(phi,thetaIndex,UniformBounds,LowerBound,thetaPositions,thetaDist,PriorDist,ModelParameters,DSGEModel,AIMData)
%
% REQUIRED INPUT: phi (vector) with all parameters to be estimated.
%
%                 thetaIndex (vector) indicating the type of distribution the parameters in theta are drawn from.
%                            If 1, then the distribution is gamma, inverse gamma, or left truncated normal.
%                            If 2, then the distribution is standardized beta. If 0, then the distribution
%                            is normal.
%
%                 UniformBounds (matrix) with the same number of rows as parameters and 2 columns. The columns
%                            are 0 1 except for the uniform and the beta distribution, where they represent
%                            the upper and lower bounds.
%
%                 LowerBound (vector) with the lower bounds for the distributions.
%
%                 thetaPositions (structure) with fields parameter and distribution. This function
%                            needs the former field to know which parameter name corresponds to
%                            position i in theta.
%
%                 thetaDist (vector) with integers 1 if gamma, 2 if beta, 3 if normal, 4 if invgamma, and 5 if
%                            truncnormal.
%
%                 PriorDist (structure) with fields "beta", "gamma", "normal", "invgamma", and "truncnormal", where
%                            the values are matrices with the parameter values for the distributions. In the case of
%                            the normal, beta, and gamma these parameter values are the mean and the standard deviation.
%                            For the invgamma it is the s and q parameters, where q is a positive integer (degrees of
%                            freedom). Finally, for the left truncated normal there's a third parameter, the lower bound.
%
%                 ModelParameters (structure) whose fields have names given by the parameter names in the
%                            DSGE model.
%
%                 DSGEModel (structure) whose fields contain all required information
%                            about the model, such as input files, variable and sample
%                            data. Actual observations are not stored in this file.
%
%                 AIMData (structure) with fields given by neq, nlag, nlead etc that is produced when running "compute_aim_data".
%                            The neq, nlag, and nlead fields are needed by AiMSolver.
%
% REQUIRED OUTPUT: logPost (scalar) the value of the log posterior at phi when we take the log of the Jacobian in the
%                            transformation theta = g(phi) into account.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: August 8, 2006.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 9-8-2006:   Fixed the logPost value when mcode~=1 or status~=0.
%
% * 25-8-2006:  Changed the order of computing the parts logLikeValue, logPrior, and logJac. Furthermore, updating
%               of the additional parameters was moved from logLikelihoodDSGE to logPosteriorPhiDSGEOptimize to make
%               sure that ModelParameters is re-evaluated in one function only. The input argument UniformBounds was
%               also added to allowe for some parameters to have a uniform prior distribution.
%
% * 28-8-2006:  Added the test if logPrior is a NaN or not.
%
% * 15-9-2006:  Checking parameters. No changes made.
%
% * 22-9-2006:  Added the drawnow command to make sure that the screen is refreshed each time the function is called.
%
% * 11-10-2006: Added a test if some of the theta parameters are infinite or NaN's. If so, the output is set to 1000000.
%
% * 24-11-2006: Changed the name from "logPosteriorPhiDSGEOptimize" to "logPosteriorPhiDSGE".
%
% * 15-3-2007:  Updated the documentation.
%
% * 29-3-2007:  Updated the documentation.
%
% * 12-11-2007: Updated the documentation.
%
% * 11-12-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 23-7-2008:  Updated the documentation.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

drawnow;
%
% transform to theta
%
theta = PhiToTheta(phi,thetaIndex,UniformBounds,LowerBound);
if (sum(isinf(theta))>=1)|(sum(isnan(theta)));
   logPost = 1000000;
   return;
end;
%
% update ModelParameters
%
ModelParameters = ThetaToModelParameters(ModelParameters,theta,thetaPositions);
%
% Update any parameters that need to be updated
%
if FileExist(DSGEModel.UpdateParameterFile)==1;
   eval(['ModelParameters = ' GetName(DSGEModel.UpdateParameterFile) '(ModelParameters);']);
end;
%
% Calculate the log-likelihood
%
[logLikeValue,mcode,status] = logLikelihoodDSGE(ModelParameters,DSGEModel,AIMData);
%
% add things up and take into account that the function will be minimzed
%
if (mcode==1)&(status==0);
   %
   % calculate the value of the log prior density at theta
   %
   logPrior = logPriorDSGE(theta,thetaDist,PriorDist,LowerBound);
   if isnan(logPrior)==0;
      %
      % calculate the log of the Jacobian
      %
      logJac = logJacobian(phi,thetaIndex,UniformBounds);
      %
      % compute the function value
      %
      logPost = -(logPrior+logJac+logLikeValue);
   else;
      logPost = 1000000;
   end;
else;
   logPost = 1000000;
end;

%
% end of logPosteriorPhiDSGE.m
%
