function AnnMatrix = Annualize(InMatrix,annual,annualscale)
% Annualize: Transforms quarterly and monthly growth rates to annual growth rates.
%
% USAGE:
%
%       AnnMatrix = Annualize(InMatrix,annual,annualscale)
%
% REQUIRED INPUT:  InMatrix (nxN) matrix.
%
%                  annual (nx1) vector with entries 1 (annual), 4 (quarterly), or 12 (annual).
%
%                  annualscale (nx1) vector with coefficients that the annualized rows will be
%                              multiplied with.
%
% REQUIRED OUTPUT: AnnMatrix (nxK) matrix where K = N, N-3, or N-11.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: January 22, 2007.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 25-1-2007:  Added the input vector annualscale, with coefficients to multiply the annaulized
%               variables with.
%
% * 9-10-2007:  Made it possible to work with NaN's.
%
% * 10-10-2007: Made sure that NaN's are correctly treated when the InMatrix is already annualized.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[n,m] = size(InMatrix);
NaNMatrix = isnan(InMatrix);
if (max(max(NaNMatrix))==1)&(max(annual)>1);
   for i=1:n;
      for j=1:m;
         if NaNMatrix(i,j)==1;
            InMatrix(i,j) = 0;
         end;
      end;
   end;
end;
if (max(annual)==4)&(n==length(annual));
   %
   % quarterly
   %
   AnnMatrix = zeros(n,m-3);
   NaNMatrix = NaNMatrix(:,4:m);
   for i=1:n;
      if annual(i)==4;
         AnnMatrix(i,:) = annualscale(i)*InMatrix(i,:)*AnnSumMatrix(m,4);
      else;
         AnnMatrix(i,:) = InMatrix(i,4:m);
      end;
   end;
   if max(max(NaNMatrix))==1;
      for i=1:n;
         for j=1:m-3;
            if NaNMatrix(i,j)==1;
               AnnMatrix(i,j) = NaN;
            end;
         end;
      end;
   end;
elseif (max(annual)==12)&(n==length(annual));
   %
   % monthly
   %
   AnnMatrix = zeros(n,m-11);
   NaNMatrix = NaNMatrix(:,12:m);
   for i=1:n;
      if annual(i)==12;
         AnnMatrix(i,:) = annualscale(i)*InMatrix(i,:)*AnnSumMatrix(m,12);
      else;
         AnnMatrix(i,:) = InMatrix(i,12:m);
      end;
   end;
   if max(max(NaNMatrix))==1;
      for i=1:n;
         for j=1:m-11;
            if NaNMatrix(i,j)==1;
               AnnMatrix(i,j) = NaN;
            end;
         end;
      end;
   end;
else;
   %
   % we already have annualized data
   %
   AnnMatrix = InMatrix;
end;

%
% end of Annualize.m
%
