function [status,MeasurementErrorStatus] = SVEExist(DSGEModel,SelectedParameters,NumDraws,TotalDraws,CurrChain,NumPaths,IsOVE)
% SVEExist: Tests if the state variable estimates have already been calculated for the DSGE model.
%
% USAGE:
%
%       [status,MeasurementErrorStatus] = SVEExist(DSGEModel,SelectedParameters,NumDraws,TotalDraws,CurrChain,NumPaths,IsOVE)
%
% REQUIRED INPUT:  DSGEModel (structure) with fields containing all the information about a DSGE Model for YADA.
%
%                  SelectedParameters (vector) of dimension NumParam which indicates which parameters can vary from
%                                     one draw to the next (1) and which parameters are fixed at posterior mode (0).
%
%                  NumDraws (integer) with the number of effective draws from the posterior distribution of the
%                                 parameters.
%
%                  TotalDraws (integer) with the number of draws from the posterior distribution of the parameters
%                                 that could have been used.
%
%                  CurrChain (integer) with the number for the current Markov chain.
%
%                  NumPaths (integer) that is positive if results for simulated data should be checked and
%                                 0 otherwise.
%
%                  IsOVE (boolean) that is 1 if observed variables have been selected and 0 if state variables
%                                 have been chosen.
%
% REQUIRED OUTPUT: status (boolean) that indicates if the state variables have already been calculated or not.
%
% OPTIONAL OUTPUT: MeasurementErrorStatus (boolean) that indicates if the model has measurement errors (1)
%                                 or not (0).
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: November 21, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 22-11-2007: Added the optional output "MeasurementErrorStatus".
%
% * 27-2-2007:  Added the input variable "NumPaths".
%
% * 1-4-2008:   Made sure that the change in the number of saved files for the simulation case is taken into
%               account. Added the input variable "IsOVE".
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
status = 0;
if nargout==2;
   MeasurementErrorStatus = 0;
end;
%
% fix the information needed to setup the file names
%
SelParamStr = strrep(num2str(SelectedParameters),' ','');
%
if NumPaths==0;
   NumFiles = ceil(NumDraws/min(NumDraws,1000));
else;
   NumFiles = ceil(NumDraws/1);
end;
%
for ThisSave=1:NumFiles;
   if NumPaths==0;
      file = [DSGEModel.OutputDirectory '\state\DSGE-SVE-' DSGEModel.NameOfModel '-' num2str(CurrChain,'%0.0f') '-' SelParamStr '-' num2str(ThisSave,'%0.0f') '-' num2str(NumFiles,'%0.0f') '.' num2str(NumDraws,'%0.0f') '-' num2str(TotalDraws,'%0.0f') '.mat'];
   else;
      if IsOVE==1;
         file = [DSGEModel.OutputDirectory '\state\DSGE-OVE-' DSGEModel.NameOfModel '-' num2str(CurrChain,'%0.0f') '-' SelParamStr '-' num2str(ThisSave,'%0.0f') '-' num2str(NumFiles,'%0.0f') '.' num2str(NumDraws,'%0.0f') '-' num2str(TotalDraws,'%0.0f') '-' num2str(NumPaths,'%0.0f') '.mat'];
      else;
         file = [DSGEModel.OutputDirectory '\state\DSGE-SVE-' DSGEModel.NameOfModel '-' num2str(CurrChain,'%0.0f') '-' SelParamStr '-' num2str(ThisSave,'%0.0f') '-' num2str(NumFiles,'%0.0f') '.' num2str(NumDraws,'%0.0f') '-' num2str(TotalDraws,'%0.0f') '-' num2str(NumPaths,'%0.0f') '.mat'];
      end;
   end;
   if FileExist(file)==0;
      return;
   else;
      if (nargout==2)&(ThisSave==1);
         SVEData = load(file);
         MeasurementErrorStatus = rank(SVEData.R);
      end;
   end;
end;
status = 1;

%
% end of SVEExist.m
%
