function [status,files] = ModestyExist(DSGEModel,controls,ModelStr)
% ModestyExist: Checks if results for the modesty statistics exist for the ModelStr type model.
%
% USAGE:
%
%       [status,files] = ModestyExist(DSGEModel,controls,ModelStr)
%
% REQUIRED INPUT:  DSGEModel (structure)  whose fields contain all required information
%                            about the model, such as input files, variable and sample
%                            data. Actual observations are not stored in this file.
%
%                  controls (structure) with handles to the YADA GUI objects.
%
%                  ModelStr (string vector) that takes on the values "DSGE" or "BVAR".
%
% REQUIRED OUTPUT: status (boolean vector) that is 1 if a modesty statistic file exists and 0
%                            otherwise.
%
%                  files (string matrix) with the full path and name of the existing files
%                            with results for the Modesty statistics.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: March 13, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-11-2007: Updated the documentation.
%
% * 21-11-2007: Updated the documentation.
%
% * 17-12-2007: The filename now taken the shock control method into account.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
status = zeros(1,10);
files = '';
%
% 1. Number of posterior draws
%
DrawsStr = get(controls.posterior.draws,'String');
NumPostDraws = str2num(StringTrim(strrep(DrawsStr(DSGEModel.PosteriorDrawsValue,:),',','')));
%
% 2. Number of parallel chains
%
ChainsStr = get(controls.posterior.chains,'String');
NumChains = str2num(StringTrim(ChainsStr(DSGEModel.ParallelChainsValue,:)));
%
% 3. Number of burnin draws per chain
%
BurninStr = get(controls.posterior.burnin,'String');
NumBurnin = str2num(strrep(StringTrim(BurninStr(DSGEModel.BurnInValue,:)),',',''));
if NumBurnin>=NumPostDraws;
   return;
end;
NumDraws = NumPostDraws-NumBurnin;
%
% check how many draws we should pick among these
%
PostDrawsUsageStr = get(controls.posterior.postdrawsusage,'String');
NumPostDrawsUsage = min(str2num(strrep(StringTrim(PostDrawsUsageStr(DSGEModel.PostDrawsUsageValue,:)),',','')),NumDraws);
%
% determine how many prediction paths to use
%
NumPredPathsStr = get(controls.forecast.predpaths,'String');
NumPredPaths = str2num(strrep(strrep(StringTrim(NumPredPathsStr(DSGEModel.NumPredPathsValue,:)),'paths',''),',',''));
%
% determine the prediction horizon
%
[MaxPH,NumPredX] = CalcMaxPredictionHorizon(DSGEModel.MaxForecastHorizon,DSGEModel.DataFrequency,DSGEModel.EndYear,DSGEModel.EndPeriod,DSGEModel.SubEndYear,DSGEModel.SubEndPeriod,size(DSGEModel.PredictedX,2));
%
if DSGEModel.ShockControlMethod==1;
   ShockMethod = 'Direct-';
else;
   ShockMethod = 'WaggonerZha-';
end;
%
% we now have all the the we need to check if the files of interest exists
%
for NChain=1:NumChains;
   file = [DSGEModel.OutputDirectory '\' ModelStr '-ModestyStatistics-' ShockMethod DSGEModel.NameOfModel '-' num2str(NChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '.' num2str(NumPostDrawsUsage,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.txt'];
   if FileExist(file)==1;
      status(1,NChain) = 1;
      files = strvcat(files,file);
   end;
end;

%
% end of ModestyExist.m
%
