function CurrINI = ReadINIFile(CurrINI)
% ReadINIFile: Read YADA.ini file (if it exists) and puts the information into the CurrINI structure
%
% USAGE:
%
%       CurrINI = ReadINIFile(CurrINI)
%
% REQUIRED INPUT:  CurrINI (structure) whose fields contain all required information
%                          about initial values of options etc.
%
% REQUIRED OUTPUT: CurrINI (structure)
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: August 2, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 3-8-2006:   Added the fields "TransformParameters", "MaximumIterationsValue", "ToleranceValue"
%
% * 7-8-2006:   Added the field "MaximizeAlgorithmValue".
%
% * 9-8-2006:   The default value for CurrINI.TransformParameters is now 1, i.e., to apply the transformation from
%               theta to phi. Added the field "StepLengthHessian", "NumberOfGridPoints", "FiniteDifferenceHessian",
%               "CheckOptimum"
%
% * 10-8-2006:  The field TransformParameters has been removed. Parameters are now always transformed by YADA.
%
% * 15-8-2006:  Added the fields "GridWidth" and "IRHorizon".
%
% * 17-8-2006:  Allowed for the possibility that "CurrINI.CurrentModelFile" is a relative path to pwd.
%
% * 18-8-2006:  Changed the StrTrim function to StringStrim.
%
% * 29-8-2006:  Added the fields "PosteriorDrawsValue", "SampleBatchValue", and "ParallelChainsValue".
%
% * 30-8-2006:  Added the fields "BurnInValue", "MHInitialScaleFactor", "MHScaleFactor", "RandomWeightValue".
%
% * 31-8-2006:  Added the fields "PriorKernel", "KernelDensityValue", and "KernelDensityEstimator".
%
% * 20-10-2006: Added the field "CheckTransformedOptimum".
%
% * 28-11-2006: The fields "PosteriorDrawsValue", "SampleBatchValue", "ParallelChainsValue", "BurnInValue",
%               "MHInitialScaleFactor", "MHScaleFactor", and "RandomWeightValue" are no loonger in the CurrentINI
%               structure. They have instead been included in the DSGEModel structure.
%
% * 7-12-2006:  The fields "MaximumIterationsValue", "ToleranceValue", "MaximizeAlgorithmValue", "StepLengthHessian",
%               "NumberOfGridPoints", "FiniteDifferenceHessian", "CheckOptimum", "CheckTransformedOptimum", "GridWidth",
%               "IRHorizon", "PriorKernel", "KernelDensityValue", and "KernelDensityEstimator" are no longer held in
%               the CurrenINI structure. They have been moved to the DSGEModel structure.
%
% * 24-1-2007:  Added the field "ModelFiles".
%
% * 7-11-2007:  Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

INIFile = [pwd '\YADA.ini'];
if FileExist(INIFile)==1;
   INITxt = ReadText(INIFile);
   INITxt = char(INITxt);
   %
   pos = LocateStr(INITxt,'CurrentModelFile = ');
   if pos>0;
      CurrINI.CurrentModelFile = StringTrim(strrep(INITxt(pos,:),'CurrentModelFile = ',''));
      %
      % check if we should prepend the string with pwd
      %
      if (isempty(CurrINI.CurrentModelFile)==0)&(strcmp(CurrINI.CurrentModelFile(1,1),'\')==1);
         CurrINI.CurrentModelFile = [pwd CurrINI.CurrentModelFile];
      end;
      if FileExist(CurrINI.CurrentModelFile)==0;
         CurrINI.CurrentModelFile = '';
      end;
   end;
   %
   pos = LocateStr(INITxt,'ModelFiles = ');
   if pos>0;
      ModelFiles = StringTrim(strrep(INITxt(pos,:),'ModelFiles = ',''));
      CurrINI.ModelFiles = Str2Matrix(ModelFiles,'$');
      %
      % check if all model files exist
      %
      ExistingFiles = [];
      for i=1:size(CurrINI.ModelFiles,1);
         if FileExist(StringTrim(CurrINI.ModelFiles(i,:)))==1;
            ExistingFiles = [ExistingFiles i];
         end;
      end;
      if isempty(ExistingFiles)==0;
         CurrINI.ModelFiles = CurrINI.ModelFiles(ExistingFiles,:);
      else;
         CurrINI.ModelFiles = CurrINI.CurrentModelFile;
      end;
   else;
      CurrINI.ModelFiles = CurrINI.CurrentModelFile;
   end;
end;

%
% end of ReadINIFile.m
%
