function CenterFillFigure(FigureHandle,FillValue,CenterValue)
% CenterFillFigure: centers and fills a matlab figure.
%
% USAGE:
%
%       CenterFillFigure(FigureHandle,FillValue,CenterValue)
%
% REQUIRED INPUTS: FigureHandle (handle) to a matlab figure.
%
%                  FillValue (integer) determines what changed to the paper size scale
%                            should be used. If "PaperPositionMode" is "auto", then
%                            this input is ignored.
%                                1 - the full scale of the paper size should be used. 
%                                2 - 1/2 of the paper size should be used.
%                                3 - 1/3 of the paper size should be used.
%                                4 - 1/4 of the paper size should be used.
%                                5 - 1/10 of the paper size should be used
%                                any other value - unchanged.
%
%                  CenterValue (integer) determines the position of the figure on the
%                            paper. If If "PaperPositionMode" is "auto", then this input
%                            if ignored.
%                               1 - center position.
%                               2 - lower left position.
%                               3 - lower right position.
%                               4 - upper left position.
%                               5 - upper right position.
%                               any other values - unchanged.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: August 11, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 23-8-2007:  Updated the documentation.
%
% * 9-11-2007:  Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(get(FigureHandle,'PaperPositionMode'),'auto')==1;
   return;
end;
%
OldPaperUnits = get(FigureHandle,'PaperUnits');
set(FigureHandle,'PaperUnits','points');
PaperSize = get(FigureHandle,'PaperSize');
PaperPosition = get(FigureHandle,'PaperPosition');
%
if FillValue==1;
   NewPaperPosition = [0 0 PaperSize];
elseif FillValue==2;
   NewPaperPosition = [(1/4)*PaperSize (1/2)*PaperSize];
elseif FillValue==3;
   NewPaperPosition = [(1/3)*PaperSize (1/3)*PaperSize];
elseif FillValue==4;
   NewPaperPosition = [(3/8)*PaperSize (1/4)*PaperSize];
elseif FillValue==5;
   NewPaperPosition = [(9/20)*PaperSize (1/10)*PaperSize];
else;
   NewPaperPosition = PaperPosition;
end;
%
PaperPosition = NewPaperPosition;
%
if CenterValue==1;
   NewLeft = (PaperSize(1)-PaperPosition(3))/2;
   NewBottom = (PaperSize(2)-PaperPosition(4))/2;
   NewPaperPosition = [NewLeft NewBottom PaperPosition(3:4)];
   %
   % check if lower left
   %
elseif CenterValue==2;
   NewPaperPosition = [0 0 PaperPosition(3:4)];
   %
   % check if lower right
   %
elseif CenterValue==3;
   NewPaperPosition = [PaperSize(1)-PaperPosition(3) 0 PaperPosition(3:4)];
   %
   % check if upper left
   %
elseif CenterValue==4;
   NewPaperPosition = [0 PaperSize(2)-PaperPosition(4) PaperPosition(3:4)];
   %
   % check if upper right
   %
elseif CenterValue==5;
   NewPaperPosition = [PaperSize(1)-PaperPosition(3) PaperSize(2)-PaperPosition(4) PaperPosition(3:4)];
end;
%
set(FigureHandle,'PaperPosition',NewPaperPosition);
set(FigureHandle,'PaperUnits',OldPaperUnits);

%
% end of CenterFillFigure.m
%
