function LineHandles = XLine(PosVec,BaseColors,CurrINI,FigureHandle)
% XLine: Draws a horizontal line with shading.
%
% USAGE:
%
%       LineHandles = XLine(PosVec,BaseColors,CurrINI,FigureHandle)
%
% REQUIRED INPUT:  PosVec (vector) of dimension 3 with the left, bottom, and width
%                         integer values for the horizontal line.
%
%                  BaseColors (vector) of dimension 2 with the first and second base
%                         color for the line (gray-scale) with min 0 and max 1.
%
%                  CurrINI (structure) with initialization information.
%
%                  FigureHandle (handle) to the figure where the line should be drawn.
%
% REQUIRED OUTPUT: LineHandles (vector) with handles to the created axes objects
%                         representing the lines.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: July 21, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 20-5-2008:  Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% get the positions
%
left = PosVec(1);
bottom = PosVec(2);
width = PosVec(3);
%
DarkColor = BaseColors(1);
LightColor = BaseColors(2);
if LightColor<DarkColor;
   check = DarkColor;
   DarkColor = LightColor;
   LightColor = check;
elseif LightColor==DarkColor;
   LightColor = 1.0;
   if DarkColor>=LightColor;
      DarkColor = 0.5;
   end;
end;
%
% set up the first and second colors
%
lightvec = [LightColor LightColor LightColor];
darkvec = [DarkColor DarkColor DarkColor];
%
if IsAlmostEqual(darkvec,CurrINI.gui_shade_color)==0;
   darkvec = CurrINI.gui_shade_color;
end;
%
if IsAlmostEqual(lightvec,CurrINI.gui_light_color)==0;
   lightvec = CurrINI.gui_light_color;
end;
%
% the x line with shading
%
h1 = axes('Parent',FigureHandle, ...
   'Units','pixels', ...
   'TickLength',[0 0], ...
   'CameraUpVector',[0 1 0], ...
   'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[left bottom width 1], ...
   'XTickLabelMode','manual', ...
   'YTickLabelMode','manual', ...
   'Tag','gui', ...
   'XColor',lightvec, ...
   'YColor',lightvec, ...
   'ZColor',lightvec);
%
h2 = axes('Parent',FigureHandle, ...
   'Units','pixels', ...
   'TickLength',[0 0], ...
   'CameraUpVector',[0 1 0], ...
   'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[left bottom+1 width 1], ...
   'XTickLabelMode','manual', ...
   'YTickLabelMode','manual', ...
   'Tag','gui', ...
   'XColor',darkvec, ...
   'YColor',darkvec, ...
   'ZColor',darkvec);
%
% check if we should add a dot!
%
if MatlabNumber>=6.0;
   h3 = axes('Parent',FigureHandle, ...
      'Units','pixels', ...
      'TickLength',[0 0], ...
      'CameraUpVector',[0 1 0], ...
      'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
      'Position',[left bottom+2 1 1], ...
      'XTickLabelMode','manual', ...
      'YTickLabelMode','manual', ...
      'Tag','gui', ...
      'XColor',get(0,'defaultuicontrolbackgroundcolor'), ...
      'YColor',get(0,'defaultuicontrolbackgroundcolor'), ...
      'ZColor',get(0,'defaultuicontrolbackgroundcolor'));
else;
   h3 = [];
end;
%
LineHandles = [h1 h2 h3];

%
% end of XLine.m
%
