function WaitHandle = WaitDLG(TextStr,ImageStr,NameStr,width,height,CurrINI,ShowCancel)
% WaitDLG: Displays a wait dialog.
%
% USAGE:
%
%       WaitHandle = WaitDLG(TextStr,ImageStr,NameStr,width,height,CurrINI,ShowCancel)
%
% REQUIRED INPUT:  TextStr (string matrix) with the text to display on the dialog.
%
%                  ImageStr (string vector) that determines which image is displayed. Valid names
%                           are "information", "question", "warning", "error". If some other
%                           string vector is given then the default logo image is chosen.
%
%                  NameStr (string vector) with the figure name.
%
%                  width (integer) with the dialog width in pixel units.
%
%                  height (integer) with the dialog height in pixel units.
%
%                  CurrINI (structure) with initialization information.
%
% OPTIONAL INPUT:  ShowCancel (boolean) that takes the value 1 if a Cancel button should be
%                          displayed on the dialog and 0 otherwise. The default value is 0.
%
% OPTIONAL OUTPUT: WaitHandle (handle) to the figure.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: December 12, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 11-5-2007:  Updated the documentation and fixed the close request function.
%
% * 15-5-2007:  Made the Cancel button 75 pixels wide rather than 80 pixels.
%
% * 9-11-2007:  Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if nargin>1;
   selector = 'init';
   if exist('ShowCancel')==0;
      ShowCancel = 0;
   end;
   %
   % get image data
   %
   ImageStructure = ReadImages;
   %
   bgcolor = get(0,'defaultuicontrolbackgroundcolor');
   %
   if strcmp(ImageStr,'information')==1;
      mylogo = ImageStructure.allimages.information_image;
   elseif strcmp(ImageStr,'question')==1;
      mylogo = ImageStructure.allimages.question_image;
   elseif strcmp(ImageStr,'warning')==1;
      mylogo =ImageStructure.allimages.warning_image;
   elseif strcmp(ImageStr,'error')==1;
      mylogo = ImageStructure.allimages.error_image;
   else;
      mylogo = ImageStructure.allimages.logo_image;
   end;
   mylogo = MirrorImage(MakeTrans(mylogo,[160 160 164],bgcolor));
else;
   WaitGUI = findobj('Type','figure','Tag','WaitDLG');
   WaitControls = get(WaitGUI,'UserData');
   selector = TextStr;
end;
%
scrsz = get(0,'ScreenSize');
%
switch selector;
   case 'init'
      WaitGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','WaitDLG.m', ...
         'MenuBar','none', ...
         'Units','pixels', ...
         'Position',[(scrsz(3)-width)/2 (scrsz(4)-height)/2 width height], ...
         'Tag','WaitDLG', ...
         'Resize','off', ...
         'Name',NameStr, ...
         'NumberTitle','off', ...
         'CloseRequestFcn','WaitDLG exit', ...
         'Visible','off', ...
         'WindowStyle','modal', ...
         'ToolBar','none');
      %
      if MatlabNumber>=7.0;
         set(WaitGUI,'DockControl','off');
      end;
      %
      imageaxes = axes('Units','pixels', ...
         'Box','on', ...
         'TickLength',[0 0], ...
         'CameraUpVector',[0 1 0], ...
         'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[10 height/2 39 39], ...
         'XTickLabelMode','manual', ...
         'YTickLabelMode','manual', ...
         'Tag','AxisLogo', ...
         'XColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'YColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'ZColor',get(0,'defaultuicontrolbackgroundcolor'));
      %
      % set the image on the axes
      %
      image('CData',mylogo,'Clipping','off','Parent',imageaxes);
      %
      WaitControls.text = uicontrol('Units','pixels', ...
            'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
            'Position',[50 50 width-52 height-54], ...
            'FontSize',CurrINI.gui_fontsize, ...
            'FontName',CurrINI.gui_fontname, ...
            'FontWeight',CurrINI.gui_fontweight, ...
            'FontAngle',CurrINI.gui_fontangle, ...
            'Style','text', ...
            'HorizontalAlignment','left', ...
            'UserData','working', ...
            'String',' ', ...
            'Tag','Text');
      %
      % need to fix TextStr string and turn it into a cell str
      %
      [out_str,pos] = textwrap(WaitControls.text,cellstr(TextStr));
      set(WaitControls.text,'String',out_str,'Position',[50 50 width-52 pos(4)]);
      if max(size(out_str))==1;
         set(WaitControls.text,'HorizontalAlignment','center');
      end;
      c_pos = 0;
      if (60+pos(4)>=90)&(height>60+pos(4));
         %clear('PredData');
         % fix the height of the dialog
         %
         set(WaitGUI,'Position',[(scrsz(3)-width)/2 (scrsz(4)-(60+pos(4)))/2 width 60+pos(4)]);
         c_pos = 60+pos(4);
      elseif (60+pos(4)<90)&(height>60+pos(4));
         set(WaitGUI,'Position',[(scrsz(3)-width)/2 (scrsz(4)-90)/2 width 90]);
         c_pos = 90;
      end;
      if c_pos>0;
         axesheight = (c_pos-90)/2;
         set(imageaxes,'Position',[10 50+axesheight 39 39]);
      end;
      %
      if ShowCancel==1;
         WaitControls.cancel = uicontrol('Units','pixels', ...
            'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
            'Position',[(width-75)/2 12 75 25], ...
            'Style','pushbutton', ...
            'String','Cancel', ...
            'FontSize',CurrINI.gui_fontsize, ...
            'FontName',CurrINI.gui_fontname, ...
            'FontWeight',CurrINI.gui_fontweight, ...
            'FontAngle',CurrINI.gui_fontangle, ...
            'Callback','set(gcbo,''UserData'',''cancel'')', ...
            'Enable','on', ...
            'UserData','', ...
            'Tag','CancelButton');
      end;
      %
      set(WaitGUI,'UserData',WaitControls,...
         'HandleVisibility','callback');
      %
      set(WaitGUI,'Visible','on');
      drawnow;
      drawnow;
   case 'exit'
      if strcmp(get(WaitControls.text,'UserData'),'done')==1;
         delete(WaitGUI);
         drawnow;
         pause(0.02);
      end;
end;
%
if nargout==1;
   WaitHandle = WaitGUI;
end;
%
if strcmp(selector,'init')==1;
   if (ShowCancel==1)&(MatlabNumber>=7);
      uicontrol(WaitControls.cancel);
   end;
end;

%
% end of WaitDLG.m
%
