function TextGUI(TextFile,NameStr,position,AxesBoxText,AxesBoxWidth,CurrINI,DisplayCopy,CloseSelf)
% TextGUI: Displays the text from a text file or a string matrix.
%
% USAGE:
%
%       TextGUI(TextFile,NameStr,position,AxesBoxText,AxesBoxWidth,CurrINI,DisplayCopy,CloseSelf)
%
% REQUIRED INPUT: TextFile (string matrix) that is either a file with text or a string
%                          matrix with text.
%
%                 NameStr (string vector) with the figure name.
%
%                 position (vector) with 4 elements giving the position of the figure
%                          in pixel units.
%
%                 AxesBoxText (string vector) with the text for the axes box.
%
%                 AxesBoxWidth (integer) width the width of the axes box.
%
%                 CurrINI (structure) with initialization information.
%
%                 DisplayCopy (boolean) which is 1 if the Copy button may be displayed
%                          and 0 otherwise.
%
% OPTIONAL INPUT: CloseSelf (boolean) which is 1 if the dialog should include code for
%                          closing itself. The default is 0.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: August 28, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 11-5-2007:  Updated the documentation.
%
% * 9-11-2007:  Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 2-4-2008:   The function can now send the TextFile to the printer.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if nargin<8;
   CloseSelf = 0;
end;
%
IsTextFile = 0;
if size(TextFile,1)==1;
   if FileExist(TextFile)==1;
      TxtStr = ReadText(TextFile);
      IsTextFile = 1;
   else;
      TxtStr = TextFile;
   end;
else;
   TxtStr = TextFile;
end;
%
if iscellstr(TxtStr)==1;
   if strcmp(StringTrim(TxtStr{max(size(TxtStr))}),'')==0;
      TxtStr{max(size(TxtStr))+1} = ' ';
   end;
else;
   if strcmp(StringTrim(TxtStr(size(TxtStr,1),:)),'')==0;
      TxtStr = strvcat(TxtStr,' ');
   end;
end;
%
boxpos = [2 2 position(3)-2 position(4)-10];
listpos = [15 54 position(3)-30 position(4)-78];
buttonpos = [position(3)-105 20 90 20];
%
textdlg = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'FileName','TextGUI.m', ...
   'MenuBar','none', ...
   'PaperUnits','points', ...
   'Units','pixels', ...
   'Position',position, ...
   'Tag','TextGUI', ...
   'DoubleBuffer','on', ...
   'Resize','on', ...
   'ResizeFcn','TextGUIResize', ...
   'Name',NameStr, ...
   'NumberTitle','off', ...
   'Visible','off', ...
   'CloseRequestFcn','CloseGUI', ...
   'ToolBar','none');
%
if MatlabNumber>=7.0;
   set(textdlg,'DockControl','off');
end;
%
% Outer axis
%
mytext.box = AxesBox(boxpos,AxesBoxText,AxesBoxWidth,[0.5 1],'on',CurrINI);
ud.boxtext = AxesBoxText;
ud.boxwidth = AxesBoxWidth;
%
% Listbox control with mytext file text
%
mytext.text = uicontrol('Units','pixels', ...
   'BackgroundColor',CurrINI.text_backgroundcolor, ...
   'ForegroundColor',CurrINI.text_color, ...
   'Position',listpos, ...
   'FontSize',CurrINI.text_fontsize, ...
   'FontName',CurrINI.text_fontname, ...
   'FontWeight',CurrINI.text_fontweight, ...
   'FontAngle',CurrINI.text_fontangle, ...
   'FontUnits',CurrINI.text_fontunits, ...
   'Style','listbox', ...
   'Enable','inactive', ...
   'UserData',ud, ...
   'Max',2, ...
   'HorizontalAlignment','center', ...
   'String',TxtStr, ...
   'Tag','TextDlg',...
   'Value',[]);
%
% Copy to clipboard button
%
if (DisplayCopy==1)&(strcmp(computer,'PCWIN')==1)&(MatlabNumber<6);
   mytext.copy = uicontrol('Units','pixels', ...
      'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
      'ForegroundColor',[0 0 0], ...
      'Position',[20 20 90 20], ...
      'String','Copy', ...
      'FontSize',CurrINI.gui_fontsize, ...
      'FontName',CurrINI.gui_fontname, ...
      'FontWeight',CurrINI.gui_fontweight, ...
      'FontAngle',CurrINI.gui_fontangle, ...
      'CallBack','TextGUICopy', ...
      'Enable','on', ...
      'Tag','Done');
end;
%
% print button
%
if IsTextFile==1;
   mytext.print = uicontrol('Units','pixels', ...
      'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
      'ForegroundColor',[0 0 0], ...
      'Position',buttonpos-[100 0 0 0], ...
      'String','Print', ...
      'FontSize',CurrINI.gui_fontsize, ...
      'FontName',CurrINI.gui_fontname, ...
      'FontWeight',CurrINI.gui_fontweight, ...
      'FontAngle',CurrINI.gui_fontangle, ...
      'CallBack','TextGUIPrint', ...
      'Enable','on', ...
      'UserData',TextFile, ...
      'Tag','Print');
end;
%
% Done button
%
mytext.done = uicontrol('Units','pixels', ...
   'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
   'ForegroundColor',[0 0 0], ...
   'Position',buttonpos, ...
   'String','Done', ...
   'FontSize',CurrINI.gui_fontsize, ...
   'FontName',CurrINI.gui_fontname, ...
   'FontWeight',CurrINI.gui_fontweight, ...
   'FontAngle',CurrINI.gui_fontangle, ...
   'CallBack','CloseGUI', ...
   'Enable','on', ...
   'UserData',CurrINI, ...
   'Tag','Done');
%
set(textdlg,'UserData',mytext, ...
   'HandleVisibility','callback');
set(textdlg,'Visible','on');
drawnow;
drawnow;
if MatlabNumber>=7;
   uicontrol(mytext.done);
end;
%
if CloseSelf==0;
   waitfor(textdlg,'waitstatus','inactive');
   delete(textdlg);
   drawnow;
   pause(0.02);
else;
   set(textdlg,'CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
   set(mytext.done,'Callback','delete(gcf); drawnow; pause(0.02);');
end;

%
% end of TextGUI.m
%
