function StatusHandle = StatusBar(CurrINI,varargin)
% StatusBar: Creates a statusbar.
%
% USAGE:
%
%       StatusHandle = StatusBar(CurrINI,varargin)
%
% REQUIRED INPUT:  CurrINI (structure) with initialization information.
%
% OPTIONAL INPUT:  Property values:
%
%                                  Positions (vector) 1x4 vector with pixel values
%
%                                  String (string vector) with text for the statusbar.
%
%                                  FontSize (integer) with the size of the font in the
%                                            statusbar. Default is 7.
%
%                                  FontName (string vector) with the name of the font.
%                                            Default is system dependent.
%
%                                  FontUnits (string vector) with the units for the
%                                            font. Default is points.
%
%                                  FontWeight (string vector) with the weight of the
%                                            font. Default is normal.
%
%                                  FontAngle (string vector) with the angle of the
%                                            font. Default is normal.
%
% OPTIONAL OUTPUT: StatusHandle (handle) to the text control of the statusbar.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: July 24, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 19-5-2008:  Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%


fig = get(0,'CurrentFigure');
if isempty(fig)==1;
   fig = gcf;
   set(fig,'Color',get(0,'defaultuicontrolbackgroundcolor'));
end;
FigureUnits = get(fig,'Units');
set(fig,'Units','pixels');
pos = get(fig,'Position');
pos = [1 1 pos(3)-1 16];
num = round(size(varargin,2)/2);
for i=1:num;
   k = 1+2*(i-1);
   if ischar(varargin{k})==1;
      if strcmp(lower(varargin{k}),'position')==1;
         pos = varargin{k+1};
         break;
      end;
   end;
end;
%
% create a box first
%
BGColor = get(0,'defaultuicontrolbackgroundcolor');
BGColor = mean(BGColor);
AxesHandles = AxesBox(pos,'',0,[0.5 BGColor BGColor 1],'on',CurrINI);
set([AxesHandles(4) AxesHandles(5)],'XColor',get(0,'defaultuicontrolbackgroundcolor'));
set([AxesHandles(4) AxesHandles(5)],'YColor',get(0,'defaultuicontrolbackgroundcolor'));
set([AxesHandles(4) AxesHandles(5)],'ZColor',get(0,'defaultuicontrolbackgroundcolor'));
%
for i=1:size(AxesHandles,2);
   if strcmp(get(AxesHandles(1,i),'Type'),'axes')==1;
      set(AxesHandles(1,i),'Tag','gui');
   end;
end;
%
% fix positions of bottom X axes and right Y axis of AxesHandles
%
set(AxesHandles(1),'Position',get(AxesHandles(1),'Position')+[-1 0 1 0]);
set(AxesHandles(8),'Position',get(AxesHandles(8),'Position')+[0 0 0 1]);
%
shadevec = 0.5*[1 1 1];
if IsAlmostEqual(shadevec,CurrINI.gui_shade_color)==0;
   shadevec = CurrINI.gui_shade_color;
end;
%
% Create a text uicontrol
%
TextHandle = uicontrol('Units','pixels', ...
   'Position',[pos(1)+4 pos(2)+1 pos(3)-8 pos(4)-2], ...
   'String','', ...
   'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
   'HorizontalAlignment','left', ...
   'FontSize',CurrINI.gui_fontsize, ...
   'FontName',CurrINI.gui_fontname, ...
   'Style','text', ...
   'Visible','on', ...
   'Tag','Statusbar');
if MatlabNumber>=7;
   set(TextHandle,'Position',[pos(1)+4 pos(2) pos(3)-8 pos(4)-2]);
end;
%
% parse input
%
prop.string = '';
prop.fontsize = 7;
prop.fontname = get(TextHandle,'FontName');
prop.fontunits = get(TextHandle,'FontUnits');
prop.fontangle = get(TextHandle,'FontAngle');
prop.fontweight = get(TextHandle,'FontWeight');
prop.handles = AxesHandles;
%
for i=1:num;
   k = 1+2*(i-1);
   if ischar(varargin{k})==1;
      if strcmp(lower(varargin{k}),'string')==1;
         set(TextHandle,'String',varargin{k+1});
         prop.string = varargin{k+1};
      elseif strcmp(lower(varargin{k}),'fontname')==1;
         set(TextHandle,'FontName',varargin{k+1});
         prop.fontname = varargin{k+1};
      elseif strcmp(lower(varargin{k}),'fontsize')==1;
         set(TextHandle,'FontSize',varargin{k+1});
         prop.fontsize = varargin{k+1};
      elseif strcmp(lower(varargin{k}),'fontunits')==1;
         set(TextHandle,'FontUnits',varargin{k+1});
         prop.units = varargin{k+1};
      elseif strcmp(lower(varargin{k}),'fontangle')==1;
         set(TextHandle,'FontAngle',varargin{k+1});
         prop.fontangle = varargin{k+1};
      elseif strcmp(lower(varargin{k}),'fontweight')==1;
         set(TextHandle,'FontWeight',varargin{k+1});
         prop.fontweight = varargin{k+1};
      end;
   end;
end;
%
set(TextHandle,'UserData',prop);
%
set(fig,'Units',FigureUnits);
%
if nargout==1;
   StatusHandle = TextHandle;
end;

%
% end of StatusBar.m
%
