function StateVarDistributionDLG(selector,SVEs,Selector,DSGEModel,CurrINI)
% StateVarDistributionDLG: Presents a dialog for plotting distribution properties of
%                          estimates of 1-step ahead forecasts, state variables, state
%                          shocks, and measurement errors from a DSGE model.
%
% USAGE:
%
%       StateVarDistributionDLG(selector,SVEs,Selector,DSGEModel,CurrINI)
%
% REQUIRED INPUT: selector (string vector) taking on the values "init", "samplebegin", "sampleend", "showgraph", and "done".
%
% OPTIONAL INPUT: SVEs (structure)  with fields that depend on Selector. For 1-step ahead forecasts the fields are:
%                      "Yhat" and "Y", where Yhat is a matrix with the mean forecasts and Y the observed data. For
%                      the state variable estimates the fields are: "Ksitt1", "Ksitt", and "KsitT". These are the
%                      mean estimates of the forecast, update, and smooth estimates of the states. For the state
%                      shock estimates fhe fields are: "etatt" and "etatT", the update and smooth estimates of the
%                      shocks. Moreover, the field "ShockNames" gives the names of the shocks that have a non-zero
%                      effect on the observed variables. For the measuremente rrors the fields are: "wtt", "wtT",
%                      the update and smooth estimates of the measurement errors. Moreover, the field "ErrorIndicator"
%                      is a vector that indicates whcih equations have a positive measurement error variance (1)
%                      and which don't (0). In addition, the vector field "Quantiles" is used, with subfields "percent"
%                      and field names determined by the Selector. For the 1-step ahead forecasts we again have
%                      "Yhat", etc. Finally, the field "DateVector" is included for all 4 cases. Required if selector
%                      is "init".
%
%                 Selector (integer) that is:
%                                  1 - if the function should work with 1-step ahead forecasts;
%                                  2 - if the function should work with state variable estimates;
%                                  3 - if the function should work with state shock estimates; and
%                                  4 - if the function should work with measurement error estimates.
%
%                 DSGEModel (structure) with fields containing all the information about a DSGE Model for YADA.
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: November 22, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 23-11-2007: Continued work on the function. Added selectable sample, state variable estimates, and
%               measurement errors.
%
% * 26-11-2007: Added the last bit of code for the 1-step ahead forecasts.
%
% * 27-11-2007: Updated the documentation.
%
% * 30-11-2007: Updated the documentation.
%
% * 2-1-2008:   Removed code using "DSGEModel.InitializeKalman".
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 11-7-2008:  Changed the ordering of colors for confidence bands.
%
% * 19-12-2208: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   if Selector==1;
      NameStr = '1-Step Ahead Forecast Distribution';
      VariableStr = 'variable';
      VariableNames = DSGEModel.VariableNames;
      AddHeight = 0;
   elseif Selector==2;
      NameStr = 'State variable Distribuitions';
      VariableStr = 'state variable';
      VariableNames = DSGEModel.StateVariableNames;
      AddHeight = 25;
   elseif Selector==3;
      NameStr = 'State Shock Distributions';
      VariableStr = 'state shock';
      VariableNames = SVEs.ShockNames;
      AddHeight = 25;
   elseif Selector==4;
      NameStr = 'Measurement Error Distributions';
      VariableStr = 'measurement error';
      DSGEModel.ErrorIndicator = SVEs.ErrorIndicator .* (1:length(SVEs.ErrorIndicator))';
      DSGEModel.ErrorIndicator = DSGEModel.ErrorIndicator(DSGEModel.ErrorIndicator>0);
      VariableNames = '';
      for i=DSGEModel.ErrorIndicator;
         VariableNames = strvcat(VariableNames,['Error in ' StringTrim(DSGEModel.VariableNames(i,:))]);
      end;
      AddHeight = 25;
   end;
   %
   % store all we need for graphics
   %
   DSGEModel.Selector = Selector;
   DSGEModel.SVEs = SVEs;
else;
   StateVarDistributionGUI = findobj('Type','figure','Tag','StateVarDistributionDLG');
   controls = get(StateVarDistributionGUI,'UserData');
   CurrINI = get(controls.variables,'UserData');
   DSGEModel = get(controls.variablestext,'UserData');
end;
switch selector;
   case 'init';
      StateVarDistributionGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','StateVarDistributionDLG.m', ...
         'MenuBar','none', ...
         'PaperUnits','points', ...
         'Units','pixels', ...
         'Position',[(CurrINI.scrsz(3)-525)/2 (CurrINI.scrsz(4)-120-AddHeight)/2 525 120+AddHeight], ...
         'Tag','StateVarDistributionDLG', ...
         'Visible','off', ...
         'Resize','off', ...
         'Name',NameStr, ...
         'NumberTitle','off', ...
         'CloseRequestFcn','StateVarDistributionDLG done', ...
         'ToolBar','none');
      %
      if MatlabNumber>=7.0;
         set(StateVarDistributionGUI,'DockControl','off');
      end;
      %
      % Outer axis
      %
      AxesBox([2 2 523 110+AddHeight],['Graphics - ' NameStr],45,[0.5 1],'on',CurrINI);
      %
      %
      % Select the sample
      %
      %
      % Check where the sample of data actually begins
      %
      [NewStartYear,NewStartPeriod] = AdjustSampleStart(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.DataFrequency,DSGEModel.KalmanFirstObservation-1);
      %
      OldDSGEModel = DSGEModel;
      DSGEModel.SubBeginYear = NewStartYear;
      DSGEModel.SubBeginPeriod = NewStartPeriod;
      %
      [FirstPeriod,LastPeriod,SubSampleStrMat] = CreateSubSample(DSGEModel);
      %
      DSGEModel = OldDSGEModel;
      controls.sampletxt = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 45+AddHeight 360 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String','Selected Sample:', ...
         'HorizontalAlignment','left', ...
         'Tag','SampleText');
      %
      % sample start and sample end
      %
      controls.samplebeg = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[210 50+AddHeight 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'String',SubSampleStrMat, ...
         'HorizontalAlignment','center', ...
         'Callback','StateVarDistributionDLG samplebegin', ...
         'Tag','SampleBeginPopup',...
         'Value',1);
      %
      controls.sampleend = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[303 50+AddHeight 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',SubSampleStrMat, ...
         'HorizontalAlignment','center', ...
         'Callback','StateVarDistributionDLG sampleend', ...
         'Tag','SampleEndPopup',...
         'Value',size(SubSampleStrMat,1));
      %
      % Select shock
      %
      controls.variablestext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 16+AddHeight 180 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'UserData',DSGEModel, ...
         'String',['Select ' VariableStr ':'], ...
         'HorizontalAlignment','left', ...
         'Tag','ObsVarText');
      %
      controls.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[210 20+AddHeight 180 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',VariableNames, ...
         'HorizontalAlignment','center', ...
         'Tag','ObsVarPopup',...
         'Value',1);
      %
      if DSGEModel.Selector==2;
         %
         % check box option when we have state shocks or measurement errors
         %
         controls.multiplot = uicontrol('Units','pixels', ...
            'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
            'Position',[24 16 341 20], ...
            'FontSize',CurrINI.gui_fontsize, ...
            'FontName',CurrINI.gui_fontname, ...
            'FontWeight',CurrINI.gui_fontweight, ...
            'FontAngle',CurrINI.gui_fontangle, ...
            'Style','checkbox', ...
            'Enable','on', ...
            'Value',1, ...
            'String','Use subplots for forecast, update, and smooth estimates', ...
            'HorizontalAlignment','left', ...
            'Tag','MultiPlotCheckBox');
      elseif DSGEModel.Selector>=3;
         %
         % check box option when we have state shocks or measurement errors
         %
         controls.multiplot = uicontrol('Units','pixels', ...
            'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
            'Position',[24 16 341 20], ...
            'FontSize',CurrINI.gui_fontsize, ...
            'FontName',CurrINI.gui_fontname, ...
            'FontWeight',CurrINI.gui_fontweight, ...
            'FontAngle',CurrINI.gui_fontangle, ...
            'Style','checkbox', ...
            'Enable','on', ...
            'Value',1, ...
            'String','Use subplots for update and smooth estimates', ...
            'HorizontalAlignment','left', ...
            'Tag','MultiPlotCheckBox');
      end;
      %
      % Create graph button
      %
      controls.show = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[420 55 90 20], ...
         'String','Display', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','StateVarDistributionDLG showgraph', ...
         'Enable','on', ...
         'Tag','Done');
      %
      %  Done button
      %
      controls.done = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[420 20 90 20], ...
         'String','Done', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','StateVarDistributionDLG done', ...
         'Enable','on', ...
         'UserData','waiting', ...
         'Tag','Done');

      set(StateVarDistributionGUI,'UserData',controls,...
         'HandleVisibility','callback');
   case 'samplebegin'
      %
      % check to make sure that the sample begins before it ends!
      %
      samplebeg = get(controls.samplebeg,'Value');
      sampleend = get(controls.sampleend,'Value');
      if sampleend<samplebeg;
         set(controls.sampleend,'Value',samplebeg);
      end;
   case 'sampleend'
      %
      % check to make sure that the sample begins before it ends
      %
      samplebeg = get(controls.samplebeg,'Value');
      sampleend = get(controls.sampleend,'Value');
      if sampleend<samplebeg;
         set(controls.samplebeg,'Value',sampleend);
      end;
   case 'showgraph'
      CurrValue = get(controls.variables,'Value');
      AllVariables = get(controls.variables,'String');
      CurrVariable = StringTrim(AllVariables(CurrValue,:));
      if CurrValue<size(AllVariables,1);
         set(controls.variables,'Value',CurrValue+1);
      else;
         set(controls.variables,'Value',1);
      end;
      samplebeg = get(controls.samplebeg,'Value');
      sampleend = get(controls.sampleend,'Value');
      DateVector = DSGEModel.SVEs.DateVector(samplebeg:sampleend);
      T = length(DateVector);
      %
      % prepare to plot
      %
      if DSGEModel.Selector==1;
         %
         % create the figure
         %
         FigHandle = figure('Visible','off', ...
                        'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                        'Tag',CurrVariable, ...
                        'Name',['1-Step Ahead Forecast - ' CurrVariable]);
         %
         SetFigureProperties(FigHandle,CurrINI);
         %
         % deal with confidence bands for the update estimates
         %
         NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
         Handles = [];
         LegendStr = '';
         %
         for nb=1:NumBounds;
            Lower = DSGEModel.SVEs.Quantiles(nb).Yhat(CurrValue,samplebeg:sampleend);
            Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Yhat(CurrValue,samplebeg:sampleend);
            PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
            ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
            ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
            set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
            hold('on');
            Handles = [Handles ConfPatch];
            LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
         end;
         set(gca,'Layer','Top');
         %
         % 1-step ahead forecast
         %
         postmean = plot(DateVector,DSGEModel.SVEs.Yhat(CurrValue,samplebeg:sampleend),'--','Color',[0 0 1],'LineWidth',2);
         hold('on');
         %
         % observed variable
         %
         obsvar = plot(DateVector,DSGEModel.SVEs.Y(CurrValue,samplebeg:sampleend),'.-','Color',[0.75 0 0],'LineWidth',2);
         hold('on');
         LegendStr = strvcat('Observed','Mean forecast (t|t-1)',LegendStr);
         Handles = [obsvar postmean Handles];
         axis('tight');
         %
         % fix the yaxis
         %
         ylim = get(gca,'YLim');
         addY = 0.1*(ylim(2)-ylim(1));
         ylim = ylim+[-addY addY];
         set(gca,'YLim',ylim);
         %
         % add a zero line if necessary
         %
         if (ylim(2)>0)&(ylim(1)<0);
            hold('on');
            plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
            kids = get(gca,'Children');
            if size(kids,1)>size(kids,2);
               kids = kids';
            end;
            kids = [kids(2) kids(3) kids(1) kids(4:length(kids))]';
            set(gca,'Children',kids);
         end;
         hold('off');
         SetAxesFonts(CurrINI,'Title',CurrVariable);
         %
         AxesObj = gca;
         %
         % add alegend
         %
         legHandle = legend(Handles,LegendStr,2);
         SetAxesFonts(CurrINI,legHandle);
         %
         SetXTickIntegers(gca);
         %
         SetToolbarImages(FigHandle,CurrINI.images);
         set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
         drawnow;
         pause(0.02);
      elseif DSGEModel.Selector==2;
         if get(controls.multiplot,'Value')==1;
            %
            % provide subplots
            %
            %
            % create the figure
            %
            FigHandle = figure('Visible','off', ...
                           'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                           'Tag',CurrVariable, ...
                           'Name',['Forecast Update and Smooth Estimates - ' CurrVariable]);
            %
            SetFigureProperties(FigHandle,CurrINI);
            %
            subplot(3,1,1);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            Handles = [];
            LegendStr = '';
            %
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).Ksitt1(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Ksitt1(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
               Handles = [Handles ConfPatch];
               LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.Ksitt1(CurrValue,samplebeg:sampleend),'-','Color',[0.75 0 0],'LineWidth',2);
            hold('on');
            LegendStr = strvcat('Mean',LegendStr);
            Handles = [postmean Handles];
            axis('tight');
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',[CurrVariable ' - Forecast estimates']);
            %
            AxesObj = gca;
            %
            % add alegend
            %
            legHandle = legend(Handles,LegendStr,2);
            SetAxesFonts(CurrINI,legHandle);
            %
            SetXTickIntegers(gca);
            ylim1 = ylim;
            %
            % the next subplot
            %
            subplot(3,1,2);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).Ksitt(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Ksitt(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.Ksitt(CurrValue,samplebeg:sampleend),'-','Color',[0.75 0 0],'LineWidth',2);
            hold('on');
            %
            axis('tight');
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',[CurrVariable ' - Update estimates']);
            %
            SetXTickIntegers(gca);
            ylim2 = ylim;
            AxesObj = [AxesObj gca];
            %
            % the last subplot
            %
            subplot(3,1,3);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).KsitT(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).KsitT(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.KsitT(CurrValue,samplebeg:sampleend),'-','Color',[0.75 0 0],'LineWidth',2);
            hold('on');
            %
            axis('tight');
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',[CurrVariable ' - Smooth estimates']);
            %
            SetXTickIntegers(gca);
            %
            % make sure that we have the same limits on the
            %
            YLim = [min([ylim(1) ylim1(1) ylim2(1)]) max([ylim(2) ylim1(2) ylim2(2)])];
            AxesObj = [AxesObj gca];
            for i=1:length(AxesObj);
               set(AxesObj(i),'YLim',YLim);
            end;
            %
            SetToolbarImages(FigHandle,CurrINI.images);
            set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
            drawnow;
            pause(0.02);
         else;
            %
            % one figure per plot
            %
            %
            % create the figure
            %
            FigHandle1 = figure('Visible','off', ...
                           'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                           'Tag',CurrVariable, ...
                           'Name',['Forecast Estimates - ' CurrVariable]);
            %
            SetFigureProperties(FigHandle1,CurrINI);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            Handles = [];
            LegendStr = '';
            %
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).Ksitt1(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Ksitt1(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
               Handles = [Handles ConfPatch];
               LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.Ksitt1(CurrValue,samplebeg:sampleend),'-','Color',[0 0 0.75],'LineWidth',2);
            hold('on');
            LegendStr = strvcat('Mean - forecast estimate',LegendStr);
            Handles = [postmean Handles];
            axis('tight');
            AxesObj = gca;
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            ylim1 = ylim;
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',CurrVariable);
            %
            % add alegend
            %
            legHandle = legend(Handles,LegendStr,2);
            SetAxesFonts(CurrINI,legHandle);
            %
            SetXTickIntegers(gca);
            %
            SetToolbarImages(FigHandle1,CurrINI.images);
            %
            % and now for the update estimates
            %
            FigHandle2 = figure('Visible','off', ...
                           'Position',[((CurrINI.scrsz(3)-650)/2)+22 ((CurrINI.scrsz(4)-500)/2)-22 650 500], ...
                           'Tag',CurrVariable, ...
                           'Name',['Update Estimates - ' CurrVariable]);
            %
            SetFigureProperties(FigHandle2,CurrINI);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            Handles = [];
            LegendStr = '';
            %
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).Ksitt(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Ksitt(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
               Handles = [Handles ConfPatch];
               LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.Ksitt(CurrValue,samplebeg:sampleend),'-','Color',[0 0.75 0],'LineWidth',2);
            hold('on');
            LegendStr = strvcat('Mean - update estimate',LegendStr);
            Handles = [postmean Handles];
            axis('tight');
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            AxesObj = [AxesObj gca];
            ylim2 = ylim;
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',CurrVariable);
            %
            % add alegend
            %
            legHandle = legend(Handles,LegendStr,2);
            SetAxesFonts(CurrINI,legHandle);
            %
            SetXTickIntegers(gca);
            %
            SetToolbarImages(FigHandle2,CurrINI.images);
            %
            % and now for the smooth estimates
            %
            FigHandle3 = figure('Visible','off', ...
                           'Position',[((CurrINI.scrsz(3)-650)/2)+44 ((CurrINI.scrsz(4)-500)/2)-44 650 500], ...
                           'Tag',CurrVariable, ...
                           'Name',['Smooth Estimates - ' CurrVariable]);
            %
            SetFigureProperties(FigHandle3,CurrINI);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            Handles = [];
            LegendStr = '';
            %
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).KsitT(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).KsitT(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
               Handles = [Handles ConfPatch];
               LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.KsitT(CurrValue,samplebeg:sampleend),'-','Color',[0.75 0 0],'LineWidth',2);
            hold('on');
            LegendStr = strvcat('Mean - smooth estimate',LegendStr);
            Handles = [postmean Handles];
            axis('tight');
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            %
            YLim = [min([ylim(1) ylim1(1) ylim2(1)]) max([ylim(2) ylim1(2) ylim2(1)])];
            AxesObj = [AxesObj gca];
            for i=1:length(AxesObj);
               set(AxesObj(i),'YLim',YLim);
            end;
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',CurrVariable);
            %
            % add alegend
            %
            legHandle = legend(Handles,LegendStr,2);
            SetAxesFonts(CurrINI,legHandle);
            %
            SetXTickIntegers(gca);
            %
            SetToolbarImages(FigHandle3,CurrINI.images);
            %
            set(FigHandle1,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
            drawnow;
            pause(0.02);
            set(FigHandle2,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
            drawnow;
            pause(0.02);
            set(FigHandle3,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
            drawnow;
            pause(0.02);
         end;
      elseif DSGEModel.Selector==3;
         if get(controls.multiplot,'Value')==1;
            %
            % provide subplots
            %
            %
            % create the figure
            %
            FigHandle = figure('Visible','off', ...
                           'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                           'Tag',CurrVariable, ...
                           'Name',['Update And Smooth Estimates - ' CurrVariable]);
            %
            SetFigureProperties(FigHandle,CurrINI);
            %
            subplot(2,1,1);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            Handles = [];
            LegendStr = '';
            %
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).etatt(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).etatt(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
               Handles = [Handles ConfPatch];
               LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.etatt(CurrValue,samplebeg:sampleend),'-','Color',[0.75 0 0],'LineWidth',2);
            hold('on');
            LegendStr = strvcat('Mean',LegendStr);
            Handles = [postmean Handles];
            axis('tight');
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',[CurrVariable ' - Update estimates']);
            %
            AxesObj = gca;
            %
            % add alegend
            %
            legHandle = legend(Handles,LegendStr,2);
            SetAxesFonts(CurrINI,legHandle);
            %
            SetXTickIntegers(gca);
            ylim1 = ylim;
            %
            % the next subplot
            %
            subplot(2,1,2);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).etatT(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).etatT(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.etatT(CurrValue,samplebeg:sampleend),'-','Color',[0.75 0 0],'LineWidth',2);
            hold('on');
            %
            axis('tight');
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',[CurrVariable ' - Smooth estimates']);
            %
            SetXTickIntegers(gca);
            %
            % make sure that we have the same limits on the
            %
            YLim = [min(ylim(1),ylim1(1)) max(ylim(2),ylim1(2))];
            AxesObj = [AxesObj gca];
            for i=1:length(AxesObj);
               set(AxesObj(i),'YLim',YLim);
            end;
            %
            SetToolbarImages(FigHandle,CurrINI.images);
            set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
            drawnow;
            pause(0.02);
         else;
            %
            % one figure per plot
            %
            %
            % create the figure
            %
            FigHandle1 = figure('Visible','off', ...
                           'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                           'Tag',CurrVariable, ...
                           'Name',['Update Estimates - ' CurrVariable]);
            %
            SetFigureProperties(FigHandle1,CurrINI);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            Handles = [];
            LegendStr = '';
            %
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).etatt(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).etatt(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
               Handles = [Handles ConfPatch];
               LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.etatt(CurrValue,samplebeg:sampleend),'-','Color',[0 0 0.75],'LineWidth',2);
            hold('on');
            LegendStr = strvcat('Mean - update estimate',LegendStr);
            Handles = [postmean Handles];
            axis('tight');
            AxesObj = gca;
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            ylim1 = ylim;
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',CurrVariable);
            %
            % add alegend
            %
            legHandle = legend(Handles,LegendStr,2);
            SetAxesFonts(CurrINI,legHandle);
            %
            SetXTickIntegers(gca);
            %
            SetToolbarImages(FigHandle1,CurrINI.images);
            %
            % and now for the smooth estimates
            %
            FigHandle2 = figure('Visible','off', ...
                           'Position',[((CurrINI.scrsz(3)-650)/2)+22 ((CurrINI.scrsz(4)-500)/2)-22 650 500], ...
                           'Tag',CurrVariable, ...
                           'Name',['Smooth Estimates - ' CurrVariable]);
            %
            SetFigureProperties(FigHandle2,CurrINI);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            Handles = [];
            LegendStr = '';
            %
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).etatT(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).etatT(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
               Handles = [Handles ConfPatch];
               LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.etatT(CurrValue,samplebeg:sampleend),'-','Color',[0.75 0 0],'LineWidth',2);
            hold('on');
            LegendStr = strvcat('Mean - smooth estimate',LegendStr);
            Handles = [postmean Handles];
            axis('tight');
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            %
            YLim = [min(ylim(1),ylim1(1)) max(ylim(2),ylim1(2))];
            AxesObj = [AxesObj gca];
            for i=1:length(AxesObj);
               set(AxesObj(i),'YLim',YLim);
            end;
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',CurrVariable);
            %
            % add alegend
            %
            legHandle = legend(Handles,LegendStr,2);
            SetAxesFonts(CurrINI,legHandle);
            %
            SetXTickIntegers(gca);
            %
            SetToolbarImages(FigHandle2,CurrINI.images);
            %
            set(FigHandle1,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
            drawnow;
            pause(0.02);
            set(FigHandle2,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
            drawnow;
            pause(0.02);
         end;
      elseif DSGEModel.Selector==4;
         %
         % create and index to translate CurrValue into a variable position in wtT etc
         %
         VarIndex = DSGEModel.SVEs.ErrorIndicator' .* (1:length(DSGEModel.SVEs.ErrorIndicator));
         VarIndex = VarIndex(VarIndex>0);
         CurrValue = VarIndex(CurrValue);
         if get(controls.multiplot,'Value')==1;
            %
            % use subplots
            %
            %
            % create the figure
            %
            FigHandle = figure('Visible','off', ...
                           'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                           'Tag',CurrVariable, ...
                           'Name',['Update And Smooth Estimates - ' CurrVariable]);
            %
            SetFigureProperties(FigHandle,CurrINI);
            %
            subplot(2,1,1);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            Handles = [];
            LegendStr = '';
            %
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).wtt(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).wtt(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
               Handles = [Handles ConfPatch];
               LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.wtt(CurrValue,samplebeg:sampleend),'-','Color',[0.75 0 0],'LineWidth',2);
            hold('on');
            LegendStr = strvcat('Mean',LegendStr);
            Handles = [postmean Handles];
            axis('tight');
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',[CurrVariable ' - Update estimates']);
            %
            AxesObj = gca;
            %
            % add alegend
            %
            legHandle = legend(Handles,LegendStr,2);
            SetAxesFonts(CurrINI,legHandle);
            %
            SetXTickIntegers(gca);
            ylim1 = ylim;
            %
            % the next subplot
            %
            subplot(2,1,2);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).wtT(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).wtT(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.wtT(CurrValue,samplebeg:sampleend),'-','Color',[0.75 0 0],'LineWidth',2);
            hold('on');
            %
            axis('tight');
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',[CurrVariable ' - Smooth estimates']);
            %
            SetXTickIntegers(gca);
            %
            % make sure that we have the same limits on the
            %
            YLim = [min(ylim(1),ylim1(1)) max(ylim(2),ylim1(2))];
            AxesObj = [AxesObj gca];
            for i=1:length(AxesObj);
               set(AxesObj(i),'YLim',YLim);
            end;
            %
            SetToolbarImages(FigHandle,CurrINI.images);
            set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
            drawnow;
            pause(0.02);
         else;
            %
            % one figure per plot
            %
            %
            % create the figure
            %
            FigHandle1 = figure('Visible','off', ...
                           'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                           'Tag',CurrVariable, ...
                           'Name',['Update Estimates - ' CurrVariable]);
            %
            SetFigureProperties(FigHandle1,CurrINI);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            Handles = [];
            LegendStr = '';
            %
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).wtt(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).wtt(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
               Handles = [Handles ConfPatch];
               LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.wtt(CurrValue,samplebeg:sampleend),'-','Color',[0 0 0.75],'LineWidth',2);
            hold('on');
            LegendStr = strvcat('Mean - update estimate',LegendStr);
            Handles = [postmean Handles];
            axis('tight');
            AxesObj = gca;
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            ylim1 = ylim;
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',CurrVariable);
            %
            % add alegend
            %
            legHandle = legend(Handles,LegendStr,2);
            SetAxesFonts(CurrINI,legHandle);
            %
            SetXTickIntegers(gca);
            %
            SetToolbarImages(FigHandle1,CurrINI.images);
            %
            % and now for the smooth estimates
            %
            FigHandle2 = figure('Visible','off', ...
                           'Position',[((CurrINI.scrsz(3)-650)/2)+22 ((CurrINI.scrsz(4)-500)/2)-22 650 500], ...
                           'Tag',CurrVariable, ...
                           'Name',['Smooth Estimates - ' CurrVariable]);
            %
            SetFigureProperties(FigHandle2,CurrINI);
            %
            % deal with confidence bands for the update estimates
            %
            NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
            Handles = [];
            LegendStr = '';
            %
            for nb=1:NumBounds;
               Lower = DSGEModel.SVEs.Quantiles(nb).wtT(CurrValue,samplebeg:sampleend);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).wtT(CurrValue,samplebeg:sampleend);
               PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
               ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
               ConfPatch = patch([DateVector fliplr(DateVector)],[Upper fliplr(Lower)],PatchColor);
               set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
               hold('on');
               Handles = [Handles ConfPatch];
               LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
            end;
            set(gca,'Layer','Top');
            %
            postmean = plot(DateVector,DSGEModel.SVEs.wtT(CurrValue,samplebeg:sampleend),'-','Color',[0.75 0 0],'LineWidth',2);
            hold('on');
            LegendStr = strvcat('Mean - smooth estimate',LegendStr);
            Handles = [postmean Handles];
            axis('tight');
            %
            % fix the yaxis
            %
            ylim = get(gca,'YLim');
            addY = 0.1*(ylim(2)-ylim(1));
            ylim = ylim+[-addY addY];
            set(gca,'YLim',ylim);
            %
            YLim = [min(ylim(1),ylim1(1)) max(ylim(2),ylim1(2))];
            AxesObj = [AxesObj gca];
            for i=1:length(AxesObj);
               set(AxesObj(i),'YLim',YLim);
            end;
            %
            % add a zero line if necessary
            %
            if (ylim(2)>0)&(ylim(1)<0);
               hold('on');
               plot([DateVector(1) DateVector(T)],[0 0],'-','Color',[0 0 0]);
               kids = get(gca,'Children');
               if size(kids,1)>size(kids,2);
                  kids = kids';
               end;
               kids = [kids(2) kids(1) kids(3:length(kids))]';
               set(gca,'Children',kids);
            end;
            hold('off');
            SetAxesFonts(CurrINI,'Title',CurrVariable);
            %
            % add alegend
            %
            legHandle = legend(Handles,LegendStr,2);
            SetAxesFonts(CurrINI,legHandle);
            %
            SetXTickIntegers(gca);
            %
            SetToolbarImages(FigHandle2,CurrINI.images);
            %
            set(FigHandle1,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
            drawnow;
            pause(0.02);
            set(FigHandle2,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
            drawnow;
            pause(0.02);
         end;
      end;
   case 'done'
      set(controls.done,'UserData','done');
      delete(StateVarDistributionGUI);
      drawnow;
      pause(0.02);
end;
if strcmp(selector,'init')==1;
   set(StateVarDistributionGUI,'Visible','on');
   drawnow;
   drawnow;
   if MatlabNumber>=7;
      uicontrol(controls.done);
   end;
   waitfor(controls.done,'UserData','done');
end;

%
% end of StateVarDistributionDLG.m
%
