function ShockGraphDLG(selector,CurrINI,DSGEModel,StateVarStructure,NameStr,ShockTypeStr)
% ShockGraphDLG: Displays graphs of state or measurement error shocks
%
% USAGE:
%
%       ShockGraphDLG(selector,CurrINI,DSGEModel,StateVarStructure,NameStr)
%
% REQUIRED INPUT: selector (string vector) taking on the values "init", "showgraph", and "done".
%
% OPTIONAL INPUT: CurrINI (structure), with initialization information.
%
%                 DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 StateVarStructure (structure) of dimension h+1, with fields Xi and Y having the responses
%                           of the state variables (Xi) and the observed variables (Y), respectively. The
%                           fields have dimension rxq and nxq respectively. The responses in the invidual
%                           variables are located in the rows, while the columns give the shocks.
%
%                 NameStr (string vector) taking on the values "Posterior Mode" or "Initial Values".
%
%                 ShockTypeStr (string vector) taking on the values "State Shock" or "Measurement Error".
%
% NOTE: When selector = "init", then all optional input arguments are required!
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: October 16, 2006
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 9-11-2006:  Fixed the X axis scale to match the date vector. Added the call to the AdjustSampleStart function.
%
% * 14-11-2006: Made sure that the StateShockNames field is adjusted for the KeepVar vector.
%
% * 17-11-2006: Changed the color and line style of the shocks in the plot.
%
% * 20-2-2007:  Added plots of the update estimates of the shocks and errors.
%
% * 27-2-2007:  Added the call "SetAxesFonts(CurrINI,legHandle)" after the legend call.
%
% * 9-11-2007:  Added the call to "SetXTickIntegers".
%
% * 22-11-2007: Updated the documentation.
%
% * 2-1-2008:   Removed code using "DSGEModel.InitializeKalman".
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 11-1-2008:  Added the possibility to select the sample.
%
% * 14-1-2008:  Fixed a bug where the DSGEModel wasn't correctly updated when setting up the sub-sample
%               string matrix.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   DSGEModel.StateVarStructure = StateVarStructure;
   DSGEModel.NameStr = NameStr;
   DSGEModel.ShockTypeStr = ShockTypeStr;
   if strcmp(ShockTypeStr,'State Shock')==1;
      ShockStr = DSGEModel.StateShockNames(StateVarStructure.KeepVar,:);
   else;
      ShockStr = StateVarStructure.wtNames;
   end;
else
   ShockGraphGUI = findobj('Type','figure','Tag','ShockGraphDLG');
   StateControls = get(ShockGraphGUI,'UserData');
   CurrINI = get(StateControls.variablespopup,'UserData');
   DSGEModel = get(StateControls.variables,'UserData');
end;

%
% Set screen size directly instead of using
% get(0,'ScreenSize')!
%

switch selector;
   case 'init';
      ShockGraphGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','ShockGraphDLG.m', ...
         'MenuBar','none', ...
         'PaperUnits','points', ...
         'Units','pixels', ...
         'Position',[(CurrINI.scrsz(3)-500)/2 (CurrINI.scrsz(4)-120)/2 500 120], ...
         'Tag','ShockGraphDLG', ...
         'Visible','off', ...
         'Resize','off', ...
         'Name',['Graph ' ShockTypeStr 's - ' NameStr], ...
         'NumberTitle','off', ...
         'CloseRequestFcn','ShockGraphDLG done', ...
         'ToolBar','none');

      if MatlabNumber>=7.0;
         set(ShockGraphGUI,'DockControl','off');
      end;
      %
      % Check where the sample of data actually begins
      %
      [NewStartYear,NewStartPeriod] = AdjustSampleStart(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.DataFrequency,DSGEModel.KalmanFirstObservation-1);
      %
      OldDSGEModel = DSGEModel;
      DSGEModel.SubBeginYear = NewStartYear;
      DSGEModel.SubBeginPeriod = NewStartPeriod;
      [FirstPeriod,LastPeriod,SubSampleStrMat] = CreateSubSample(DSGEModel);
      %
      DSGEModel = OldDSGEModel;
      %
      % Outer axis
      %
      AxesBox([2 2 498 110],'Graphics',45,[0.5 1],'on',CurrINI);
      %
      % Sample selection controls
      %
      StateControls.sampletxt = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 50 360 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String','Selected Sample:', ...
         'HorizontalAlignment','left', ...
         'Tag','SampleText');
      %
      % sample start and sample end
      %
      StateControls.samplebeg = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[185 55 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',SubSampleStrMat, ...
         'HorizontalAlignment','center', ...
         'Callback','ShockGraphDLG samplebegin', ...
         'Tag','SampleBeginPopup',...
         'Value',1);
      %
      StateControls.sampleend = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[278 55 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',SubSampleStrMat, ...
         'HorizontalAlignment','center', ...
         'Callback','ShockGraphDLG sampleend', ...
         'Tag','SampleEndPopup',...
         'Value',size(SubSampleStrMat,1));
      %
      % Select variable
      %
      StateControls.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 16 160 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'UserData',DSGEModel, ...
         'String',['Select ' lower(ShockTypeStr) ':'], ...
         'HorizontalAlignment','left', ...
         'Tag','VariablesText');
      %
      StateControls.variablespopup = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[185 20 180 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',ShockStr, ...
         'HorizontalAlignment','center', ...
         'Tag','VariablePopup',...
         'Value',1);
      %
      % Create graph button
      %
      StateControls.show = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 55 90 20], ...
         'String','Display', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','ShockGraphDLG showgraph', ...
         'Enable','on', ...
         'Tag','Done');
      %
      %  Done button
      %
      StateControls.done = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 20 90 20], ...
         'String','Done', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','ShockGraphDLG done', ...
         'Enable','on', ...
         'UserData','waiting', ...
         'Tag','Done');

      set(ShockGraphGUI,'UserData',StateControls,...
         'HandleVisibility','callback');
   case 'samplebegin'
      %
      % check to make sure that the sample begins before it ends!
      %
      samplebeg = get(StateControls.samplebeg,'Value');
      sampleend = get(StateControls.sampleend,'Value');
      if sampleend<samplebeg;
         set(StateControls.sampleend,'Value',samplebeg);
      end;
   case 'sampleend'
      %
      % check to make sure that the sample begins before it ends
      %
      samplebeg = get(StateControls.samplebeg,'Value');
      sampleend = get(StateControls.sampleend,'Value');
      if sampleend<samplebeg;
         set(StateControls.samplebeg,'Value',sampleend);
      end;
   case 'showgraph'
      VarValue = get(StateControls.variablespopup,'Value');
      VarString = get(StateControls.variablespopup,'String');
      CurrVariable = StringTrim(VarString(VarValue,:));
      if VarValue<size(VarString,1);
         set(StateControls.variablespopup,'Value',VarValue+1);
      else;
         set(StateControls.variablespopup,'Value',1);
      end;
      %
      % create a vector with dates
      %
      DateVector = MakeDateVector(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.SubEndYear,DSGEModel.SubEndPeriod,DSGEModel.DataFrequency);
      %
      % get the selected sample
      %
      StartSample = get(StateControls.samplebeg,'Value');
      EndSample = get(StateControls.sampleend,'Value');
      %
      % create the figure
      %
      FigHandle =figure('Visible','off', ...
                'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                'Tag',CurrVariable, ...
                'Name',['Graph of ' CurrVariable]);
      %
      SetFigureProperties(FigHandle,CurrINI);
      %
      % Make sure that DateVector is not longer than the number of periods for the shocks
      %
      if length(DateVector)>length(DSGEModel.StateVarStructure.etatT);
         DateVector = DateVector(1,length(DateVector)-length(DSGEModel.StateVarStructure.etatT)+1:length(DateVector));
      end;
      %
      if strcmp(DSGEModel.ShockTypeStr,'State Shock')==1;
         %
         % we're plotting the state shocks
         %
         UpdateHandle = plot(DateVector(StartSample:EndSample),DSGEModel.StateVarStructure.etatt(VarValue,StartSample:EndSample),'-.','LineWidth',2,'Color',[0.5 0.5 0.5]);
         hold('on');
         SmoothHandle = plot(DateVector(StartSample:EndSample),DSGEModel.StateVarStructure.etatT(VarValue,StartSample:EndSample),':','LineWidth',2,'Color',[1 0 0]);
      else;
         %
         % we're plotting the measurement errors
         %
         UpdateHandle = plot(DateVector(StartSample:EndSample),DSGEModel.StateVarStructure.wtt(VarValue,StartSample:EndSample),'-.','LineWidth',2,'Color',[0.5 0.5 0.5]);
         hold('on');
         SmoothHandle = plot(DateVector(StartSample:EndSample),DSGEModel.StateVarStructure.wtT(VarValue,StartSample:EndSample),':','LineWidth',2,'Color',[1 0 0]);
      end;
      hold('on');
      ylim = get(gca,'YLim');
      kids = get(gca,'Children');
      if size(kids,2)>size(kids,1);
         kids = kids';
      end;
      ZeroHandle = plot([DateVector(StartSample) DateVector(EndSample)],[0 0],'-','Color',[0 0 0]);
      set(gca,'Children',[ZeroHandle;kids]);
      set(gca,'XLim',[DateVector(StartSample) DateVector(EndSample)]);
      %
      SetAxesFonts(CurrINI,'Title',[CurrVariable ' projections at ' lower(DSGEModel.NameStr)]);
      %
      % add the legend
      %
      legHandle = legend([UpdateHandle SmoothHandle],strvcat('Update (t|t)','Smoothing (t|T)'));
      SetAxesFonts(CurrINI,legHandle);
      %
      SetXTickIntegers(gca);
      %
      % Fix the toolbar
      %
      SetToolbarImages(FigHandle,CurrINI.images);
      set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
      drawnow;
      pause(0.02);
   case 'done'
      set(StateControls.done,'UserData','done');
      delete(ShockGraphGUI);
      drawnow;
      pause(0.02);
end;
%
if strcmp(selector,'init')==1;
   set(ShockGraphGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(StateControls.done);
   end;
   waitfor(StateControls.done,'UserData','done');
end;

%
% end of ShockGraphDLG.m
%
