function [action,positions,ShockGroupName,ShockGroupColor,ShockColorStart,ShockColorEnd] = SetShockGroupsDLG(selector,ShockGroupNumber,AllVars,positions,ShockGroupName,ShockGroupColor,ShockColorStart,ShockColorEnd,CurrINI)
% SetShockGroupsDLG: Dialog for either selecting the state shocks or the state variables.
%
% USAGE:
%
%       [action,positions,ShockGroupName,ShockGroupColor,ShockColorStart,ShockColorEnd] = SetShockGroupsDLG(selector,ShockGroupNumber,AllVars,positions,ShockGroupName,ShockGroupColor,ShockColorStart,ShockColorEnd,CurrINI)
%
% REQUIRED INPUT:  selector (string vector), with the possible values "init", "checkname", "quit"
%
% OPTIONAL INPUT:  ShockGroupNumber (integer), representing the group number.
%
%                  AllVars (string matrix), with all the possible state variable and state shock names. Given
%                          by the AiM output.
%
%                  positions (vector), with the selected entries from AllVars.
%
%                  ShockGroupName (string), with the current name of the shock group.
%
%                  ShockGroupColor (1x3 vector) with a color code for the shock group.
%
%                  ShockColorStart (1x3 vector) with a color code for start color of shocks in
%                         and individual shock group.
%
%                  ShockColorEnd (1x3 vector) with a color code for end color of shocks in
%                         and individual shock group.
%
%                  CurrINI (structure), with initialization information.
%
% Optional OUTPUT: action (string vector) taking the value "OK" or "Cancel". Depends on which button was
%                         clicked on.
%
%                  positions (vector), with the selected entries from AllVars.
%
%                  ShockGroupName (string vector). with the new shock group name.
%
%                  ShockGroupColor (1x3 vector) with a color code for the shock group.
%
%                  ShockColorStart (1x3 vector) with a color code for start color of shocks in
%                         and individual shock group.
%
%                  ShockColorEnd (1x3 vector) with a color code for end color of shocks in
%                         and individual shock group.
%
% NOTE: If selector = "init", then  all optional inputs and outputs are required.
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: December 4, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 5-12-2006:  Replaced spaces with _.
%
% * 5-1-2007:   Added the new input "ShockGroupColor" and the new output "ShockGroupColor".
%
% * 9-11-2007:  Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 31-1-2008:  Added the input variables "ShockColorStart" and "ShockColorEnd".
%
% * 21-5-2008:  Removed unnecessary callback from checkbox.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   SV.AllVars = AllVars;
   SV.positions = positions;
   SV.ShockGroupNumber = ShockGroupNumber;
   SV.ShockColorStart = ShockColorStart;
   SV.ShockColorEnd = ShockColorEnd;
   scrsz = get(0,'ScreenSize');
   %
   % try to determine the width and the height
   %
   gui_cols = min(ceil(sqrt(size(AllVars,1))),9);
   if gui_cols<=min(4,size(AllVars,1));
      gui_cols = min(4,size(AllVars,1));
   end;
   gui_rows = ceil(size(AllVars,1)/gui_cols);
   width = (100*gui_cols)+50+(5*(gui_cols-1));
   if width<640;
      width = 640;
   end;
   height = min(scrsz(4)-100,110+(20*gui_rows)+(5*(gui_rows-1)));
else;
   SetShockGroupsGUI = findobj('Type','figure','Tag','SetShockGroupsDLG');
   controls = get(SetShockGroupsGUI,'UserData');
   SV = get(controls.cancel,'UserData');
end;
%
switch selector
   case 'init'
      SetShockGroupsGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','SetShockGroupsDLG.m', ...
         'MenuBar','none', ...
         'Units','pixels', ...
         'Position',[(scrsz(3)-width)/2 (scrsz(4)-height)/2 width height], ...
         'Tag','SetShockGroupsDLG', ...
         'Resize','off', ...
         'Visible','off', ...
         'Name',['Select Shocks - Group ' num2str(ShockGroupNumber)], ...
         'NumberTitle','off', ...
         'WindowStyle','modal', ...
         'CloseRequestFcn','SetShockGroupsDLG quit', ...
         'ToolBar','none');
      %
      if MatlabNumber>=7.0;
         set(SetShockGroupsGUI,'DockControl','off');
      end;
      %
      AxesBox([2 2 width-2 height-10],'Shock Group Selection',100,[0.5 1],'on',CurrINI);
      %
      % checkboxes with all variables from the AiM output
      %
      chkbx = [];
      posmat = zeros(size(AllVars,1),2);
      num = 0;
      for i=1:gui_rows;
         for j=1:gui_cols;
            num = num+1;
            if num>size(AllVars,1);
               break;
            end;
            pos = [20+(105*(j-1)) height-35-(20*i)];
            new_chkbx = uicontrol('Units','pixels', ...
               'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
               'Position',[pos 100 20], ...
               'FontSize',CurrINI.gui_fontsize, ...
               'FontName',CurrINI.gui_fontname, ...
               'FontWeight',CurrINI.gui_fontweight, ...
               'FontAngle',CurrINI.gui_fontangle, ...
               'Style','checkbox', ...
               'Visible','on', ...
               'HorizontalAlignment','left', ...
               'String',StringTrim(AllVars(num,:)), ...
               'Value',positions(num), ...
               'Enable','on', ...
               'Tag','UICheckBox');
            chkbx = [chkbx new_chkbx];
            posmat(num,:) = pos;
         end;
      end;
      SV.posmat = posmat;
      SV.chkbx = chkbx;
      %
      % edit dialog with the name of the group
      %
      controls.groupnametext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[20 40 width-210 20], ...
         'String','OK', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Style','text', ...
         'String','Set shock group name:', ...
         'HorizontalAlignment','left', ...
         'Tag','GroupNameText');
      %
      controls.groupname = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[20 20 width-450 25], ...
         'String','OK', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','edit', ...
         'String',strrep(ShockGroupName,'_',' '), ...
         'UserData',strrep(ShockGroupName,'_',' '), ...
         'Enable','on', ...
         'HorizontalAlignment','left', ...
         'Callback','SetShockGroupsDLG checkname', ...
         'Tag','GroupNameEdit');
      %
      % button to selct a new start color
      %
      controls.startcolor = uicontrol('Units','pixels', ...
         'BackgroundColor',ShockColorStart, ...
         'ForegroundColor',1-ShockColorStart, ...
         'Position',[width-415 20 75 25], ...
         'String','Start Color', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','SetShockGroupsDLG startcolor', ...
         'UserData',ShockColorStart, ...
         'Tag','ButtonStartColor');
      %
      % button to select a new group color
      %
      controls.color = uicontrol('Units','pixels', ...
         'BackgroundColor',ShockGroupColor, ...
         'ForegroundColor',1-ShockGroupColor, ...
         'Position',[width-335 20 75 25], ...
         'String','Group Color', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','SetShockGroupsDLG color', ...
         'UserData',ShockGroupColor, ...
         'Tag','ButtonColor');
      %
      % button to selct a new end color
      %
      controls.endcolor = uicontrol('Units','pixels', ...
         'BackgroundColor',ShockColorEnd, ...
         'ForegroundColor',1-ShockColorEnd, ...
         'Position',[width-255 20 75 25], ...
         'String','End Color', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','SetShockGroupsDLG endcolor', ...
         'UserData',ShockColorEnd, ...
         'Tag','ButtonEndColor');
      %
      if (MatlabNumber>=7)&(IsWinXP==1);
         %
         % set the CData property
         %
         Cdat(:,:,1) = ones(25,75)*ShockGroupColor(1);
         Cdat(:,:,2) = ones(25,75)*ShockGroupColor(2);
         Cdat(:,:,3) = ones(25,75)*ShockGroupColor(3);
         set(controls.color,'CData',Cdat);
         %
         Cdat(:,:,1) = ones(25,75)*ShockColorStart(1);
         Cdat(:,:,2) = ones(25,75)*ShockColorStart(2);
         Cdat(:,:,3) = ones(25,75)*ShockColorStart(3);
         set(controls.startcolor,'CData',Cdat);
         %
         Cdat(:,:,1) = ones(25,75)*ShockColorEnd(1);
         Cdat(:,:,2) = ones(25,75)*ShockColorEnd(2);
         Cdat(:,:,3) = ones(25,75)*ShockColorEnd(3);
         set(controls.endcolor,'CData',Cdat);
      end;
      %
      % Two buttons to select action
      %
      controls.ok = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-175 20 75 25], ...
         'String','OK', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','SetShockGroupsDLG quit', ...
         'UserData','waiting', ...
         'Tag','ButtonOK');
      %
      controls.cancel = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-95 20 75 25], ...
         'String','Cancel', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'UserData',SV, ...
         'Callback','SetShockGroupsDLG quit', ...
         'Tag','ButtonCancel');
      %
      set(SetShockGroupsGUI,'UserData',controls);
   case 'color'
      %
      % change the color
      %
      CurrColor = get(controls.color,'BackgroundColor');
      NewColor = uisetcolor(CurrColor,'Select Group Color');
      if (isnumeric(NewColor)==1)&(length(NewColor)==3);
         set(controls.color,'BackgroundColor',NewColor,'ForegroundColor',1-NewColor);
         if (MatlabNumber>=7)&(IsWinXP==1);
            %
            % set the CData property
            %
            Cdat(:,:,1) = ones(25,75)*NewColor(1);
            Cdat(:,:,2) = ones(25,75)*NewColor(2);
            Cdat(:,:,3) = ones(25,75)*NewColor(3);
            set(controls.color,'CData',Cdat);
         end;
      end;
   case 'endcolor'
      %
      % change the color
      %
      CurrColor = get(controls.endcolor,'BackgroundColor');
      NewColor = uisetcolor(CurrColor,'Select End Color');
      if (isnumeric(NewColor)==1)&(length(NewColor)==3);
         set(controls.endcolor,'BackgroundColor',NewColor,'ForegroundColor',1-NewColor);
         if (MatlabNumber>=7)&(IsWinXP==1);
            %
            % set the CData property
            %
            Cdat(:,:,1) = ones(25,75)*NewColor(1);
            Cdat(:,:,2) = ones(25,75)*NewColor(2);
            Cdat(:,:,3) = ones(25,75)*NewColor(3);
            set(controls.endcolor,'CData',Cdat);
         end;
      end;
   case 'startcolor'
      %
      % change the color
      %
      CurrColor = get(controls.startcolor,'BackgroundColor');
      NewColor = uisetcolor(CurrColor,'Select Start Color');
      if (isnumeric(NewColor)==1)&(length(NewColor)==3);
         set(controls.startcolor,'BackgroundColor',NewColor,'ForegroundColor',1-NewColor);
         if (MatlabNumber>=7)&(IsWinXP==1);
            %
            % set the CData property
            %
            Cdat(:,:,1) = ones(25,75)*NewColor(1);
            Cdat(:,:,2) = ones(25,75)*NewColor(2);
            Cdat(:,:,3) = ones(25,75)*NewColor(3);
            set(controls.startcolor,'CData',Cdat);
         end;
      end;
   case 'checkname'
      %
      % checks to make sure the shock group name is a valid text string. $ signs are not allowed!
      %
      newname = StringTrim(strrep(get(controls.groupname,'String'),'$',''));
      if isempty(newname)==1;
         set(controls.groupname,'String',get(controls.groupname,'UserData'));
      else;
         set(controls.groupname,'String',newname,'UserData',newname);
      end;
   case 'quit'
      set(controls.ok,'UserData','done');
end;

if strcmp(selector,'init')==1;
   set(SetShockGroupsGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(controls.ok);
   end;
   %
   waitfor(controls.ok,'UserData','done');
   if (isempty(get(SetShockGroupsGUI,'CurrentObject'))==1)|(strcmp(get(get(SetShockGroupsGUI,'CurrentObject'),'Type'),'uicontrol')==0);
      action = 'Cancel';
   else;
      action = get(get(SetShockGroupsGUI,'CurrentObject'),'String');
      if (isempty(action)==1)|(strcmp(action,'OK')==0);
         action = 'Cancel';
      end;
   end;
   SV = get(controls.cancel,'UserData');
   %
   % determine output
   %
   positions = [];
   for i=1:length(SV.chkbx);
      positions = [positions get(SV.chkbx(i),'Value')];
   end;
   ShockGroupName = strrep(get(controls.groupname,'UserData'),' ','_');
   if strcmp(lower(action),'ok')==1;
      ShockGroupColor = get(controls.color,'BackgroundColor');
      ShockColorEnd = get(controls.endcolor,'BackgroundColor');
      ShockColorStart = get(controls.startcolor,'BackgroundColor');
   else;
      ShockGroupColor = get(controls.color,'UserData');
      ShockColorEnd = get(controls.endcolor,'UserData');
      ShockColorStart = get(controls.startcolor,'UserData');
   end;
   delete(SetShockGroupsGUI);
   drawnow;
   pause(0.02);
end;

%
% end of SetShockGroupsDLG.m
%
