function [action,positions] = SelectCondVarShockDLG(selector,vartype,infotype,AllVars,positions,CurrINI)
% SelectCondVarShockDLG: Provides a dialog for selecting which conditional variables or shocks
%                        to use in conditional forecasting.
%
% USAGE:
%
%       [action,positions] = SelectCondVarShockDLG(selector,vartype,infotype,AllVars,positions,CurrINI)
%
% REQUIRED INPUT:  selector (string vector), with the possible values "init", "slider", "quit"
%
% OPTIONAL INPUT:  vartype (string vector), taking on the value "Variables", "Shocks", or "Parameters".
%
%                  infotype (string vector), taking on the value "Conditioning" or "Scenario".
%
%                  AllVars (string matrix), with all the possible names of the conditioning variables or
%                          shocks.
%
%                  positions (vector), with row number for the positions in AllVars of the vartype.
%
%                  CurrINI (structure), with initialization information.
%
% Optional OUTPUT: action (string vector) taking the value "OK" or "Cancel". Depends on which button was
%                         clicked on.
%
%                  positions (vector), with row number for the positions in AllVars of the vartype.
%
% NOTE: If selector = "init", then  all optional inputs and outputs are required.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: March 8, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 25-4-2007:  Added the "infotype" input.
%
% * 2-5-2007:   Added the option "Parameters" to the "vartype" input.
%
% * 9-11-2007:  Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 21-5-2008:  Removed unnecessary callback from checkbox.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   SV.AllVars = AllVars;
   SV.positions = positions;
   scrsz = get(0,'ScreenSize');
   %
   % try to determine the width and the height
   %
   gui_cols = min(ceil(sqrt(size(AllVars,1))),9);
   if gui_cols<=min(4,size(AllVars,1));
      gui_cols = min(4,size(AllVars,1));
   end;
   gui_rows = ceil(size(AllVars,1)/gui_cols);
   width = (100*gui_cols)+50+(5*(gui_cols-1));
   if width<300;
      width = 300;
   end;
   height = min(scrsz(4)-100,100+(20*gui_rows)+(5*(gui_rows-1)));
else;
   SelectCondVarShockGUI = findobj('Type','figure','Tag','SelectCondVarShockDLG');
   controls = get(SelectCondVarShockGUI,'UserData');
   SV = get(controls.cancel,'UserData');
end;

switch selector
   case 'init'
      SelectCondVarShockGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','SelectCondVarShockDLG.m', ...
         'MenuBar','none', ...
         'Units','pixels', ...
         'Position',[(scrsz(3)-width)/2 (scrsz(4)-height)/2 width height], ...
         'Tag','SelectCondVarShockDLG', ...
         'Resize','off', ...
         'Visible','off', ...
         'Name',['Select ' infotype ' ' vartype], ...
         'NumberTitle','off', ...
         'WindowStyle','modal', ...
         'CloseRequestFcn','SelectCondVarShockDLG quit', ...
         'ToolBar','none');
      %
      if MatlabNumber>=7.0;
         set(SelectCondVarShockGUI,'DockControl','off');
      end;
      %
      AxesBox([2 2 width-2 height-10],[infotype ' ' vartype ' Selection'],100,[0.5 1],'on',CurrINI);
      %
      % checkboxes with all variables from the AiM output
      %
      chkbx = [];
      posmat = zeros(size(AllVars,1),2);
      num = 0;
      for i=1:gui_rows;
         for j=1:gui_cols;
            num = num+1;
            if num>size(AllVars,1);
               break;
            end;
            pos = [20+(105*(j-1)) height-35-(20*i)];
            new_chkbx = uicontrol('Units','pixels', ...
               'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
               'Position',[pos 100 20], ...
               'FontSize',CurrINI.gui_fontsize, ...
               'FontName',CurrINI.gui_fontname, ...
               'FontWeight',CurrINI.gui_fontweight, ...
               'FontAngle',CurrINI.gui_fontangle, ...
               'Style','checkbox', ...
               'Visible','on', ...
               'HorizontalAlignment','left', ...
               'String',StringTrim(AllVars(num,:)), ...
               'Value',sum(num==positions), ...
               'Enable','on', ...
               'Tag','UICheckBox');
            chkbx = [chkbx new_chkbx];
            posmat(num,:) = pos;
         end;
      end;
      SV.posmat = posmat;
      SV.chkbx = chkbx;
      %
      % Two buttons to select action
      %
      controls.ok = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-175 20 75 25], ...
         'String','OK', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','SelectCondVarShockDLG quit', ...
         'UserData','waiting', ...
         'Tag','ButtonOK');
      %
      controls.cancel = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-95 20 75 25], ...
         'String','Cancel', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'UserData',SV, ...
         'Callback','SelectCondVarShockDLG quit', ...
         'Tag','ButtonCancel');
      %
      set(SelectCondVarShockGUI,'UserData',controls);
   case 'slider'
      %
      % in case the height of the dialog is inadequate to show all controls.
      % This requires more than 200 variables in the AiM model!
      %
   case 'quit'
      set(controls.ok,'UserData','done');
end;

if strcmp(selector,'init')==1;
   set(SelectCondVarShockGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(controls.ok);
   end;
   %
   waitfor(controls.ok,'UserData','done');
   if (isempty(get(SelectCondVarShockGUI,'CurrentObject'))==1)|(strcmp(get(get(SelectCondVarShockGUI,'CurrentObject'),'Type'),'uicontrol')==0);
      action = 'Cancel';
   else;
      action = get(get(SelectCondVarShockGUI,'CurrentObject'),'String');
      if (isempty(action)==1)|(strcmp(action,'OK')==0);
         action = 'Cancel';
      end;
   end;
   SV = get(controls.cancel,'UserData');
   %
   % determine output
   %
   positions = [];
   for i=1:length(SV.chkbx);
      if get(SV.chkbx(i),'Value')==1;
         positions = [positions i];
      end;
   end;
   delete(SelectCondVarShockGUI);
   drawnow;
   pause(0.02);
end;

%
% end of SelectCondVarShockDLG.m
%
