function ScatterPlotPosteriorDraws(selector,ParamSample,LogPost,ParameterNames,ParamMode,CurrChain,TypeStr,DSGEModel,CurrINI)
% ScatterPlotPosteriorDraws: Displays a dialog for selecting 2 parameters to use for computing scatter
%                            plots of 2 parameters.
%
% USAGE:
%
%       ScatterPlotPosteriorDraws(selector,ParamSample,LogPost,ParameterNames,ParamMode,CurrChain,TypeStr,DSGEModel,CurrINI)
%
% REQUIRED INPUT: selector (string vector) that takes on the values  "init", "showgraph",
%                          "checkparams", and "done".
%
% OPTIONAL INPUT: ParamSample (matrix) of dimension NumDrawsxNumParam with all or a subset the posterior
%                          draws of the original parameters.
%
%                 LogPost (vector) with the log posterior values for the draws.
%
%                 ParameterNames (string matrix) with the names of the parameters.
%
%                 ParamMode (vector) with the posterior mode estimates of the parameters.
%
%                 CurrChain (integer) with the number for the used sampling chain.
%
%                 TypeStr (string vector) that takes the values "Original" or "Transformed".
%
%                 DSGEModel (structure)  whose fields contain all required information
%                          about the model, such as input files, variable and sample
%                          data. Actual observations are not stored in this file.
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: April 4, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 5-4-2007:   Added the input "ParamMode" so that the posterior mode estimates can be included in the scatter plots.
%
% * 16-4-2007:  Made sure that the mode lines appear above the scatter plot.
%
% * 17-4-2007:  Updated the documentation.
%
% * 9-11-2007:  Updated the documentation.
%
% * 2-1-2008:   Removed code using "DSGEModel.InitializeKalman".
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   DSGEModel.ParamSample = ParamSample;
   DSGEModel.ParamMode = ParamMode;
   DSGEModel.LogPost = round((63/max(LogPost-min(LogPost)))*(LogPost-min(LogPost)))+1;
   DSGEModel.CurrChain = CurrChain;
   DSGEModel.TypeStr = TypeStr;
else;
   ScatterPostDrawsGUI = findobj('Type','figure','Tag','ScatterPlotPosteriorDraws');
   ScatterControls = get(ScatterPostDrawsGUI,'UserData');
   CurrINI = get(ScatterControls.parameter2popup,'UserData');
   DSGEModel = get(ScatterControls.parameter2text,'UserData');
end;
%
% Set screen size directly instead of using
% get(0,'ScreenSize')!
%
switch selector;
   case 'init';
      ScatterPostDrawsGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','ScatterPlotPosteriorDraws.m', ...
         'MenuBar','none', ...
         'PaperUnits','points', ...
         'Units','pixels', ...
         'Position',[(CurrINI.scrsz(3)-500)/2 (CurrINI.scrsz(4)-180)/2 500 180], ...
         'Tag','ScatterPlotPosteriorDraws', ...
         'Visible','off', ...
         'Resize','off', ...
         'Name',['Scatter-Plot ' num2str(length(LogPost),'%0.0f') ' Posterior Draws - ' DSGEModel.TypeStr ' Data'], ...
         'NumberTitle','off', ...
         'CloseRequestFcn','ScatterPlotPosteriorDraws done', ...
         'ToolBar','none');

      if MatlabNumber>=7.0;
         set(ScatterPostDrawsGUI,'DockControl','off');
      end;
      %
      % Check where the sample of data actually begins
      %
      [NewStartYear,NewStartPeriod] = AdjustSampleStart(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.DataFrequency,DSGEModel.KalmanFirstObservation-1);
      %
      % Outer axis
      %
      AxesBox([2 2 498 170],'Graphics',45,[0.5 1],'on',CurrINI);
      %
      % sample infortmation
      %
      ScatterControls.sample = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 110 360 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String',['Selected Sample: ' NewStartYear ':' NewStartPeriod ' - ' DSGEModel.SubEndYear ':' DSGEModel.SubEndPeriod], ...
         'HorizontalAlignment','left', ...
         'Tag','SampleText');
      %
      % select parameter 1
      %
      ScatterControls.parameter1text = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 75 180 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String','Select parameter 1:', ...
         'HorizontalAlignment','left', ...
         'Tag','Parameter1Text');
      %
      ScatterControls.parameter1popup = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[210 80 155 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',ParameterNames, ...
         'Callback','ScatterPlotPosteriorDraws checkparams', ...
         'HorizontalAlignment','center', ...
         'Tag','Parameter1Popup',...
         'Value',1);
      %
      % Select parameter 2
      %
      ScatterControls.parameter2text = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 45 180 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'UserData',DSGEModel, ...
         'String','Select parameter 2:', ...
         'HorizontalAlignment','left', ...
         'Tag','ObsVarText');
      %
      ScatterControls.parameter2popup = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[210 50 155 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',ParameterNames, ...
         'Callback','ScatterPlotPosteriorDraws checkparams', ...
         'HorizontalAlignment','center', ...
         'Tag','Parameter2Popup',...
         'Value',2);
      %
      % check if multiple colors should be used
      %
      ScatterControls.Colors = uicontrol('Units','pixels', ...
            'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
            'Position',[24 20 340 20], ...
            'FontSize',CurrINI.gui_fontsize, ...
            'FontName',CurrINI.gui_fontname, ...
            'FontWeight',CurrINI.gui_fontweight, ...
            'FontAngle',CurrINI.gui_fontangle, ...
            'Style','checkbox', ...
            'Visible','on', ...
            'HorizontalAlignment','left', ...
            'String','Use colors based on log posterior value', ...
            'Value',0, ...
            'Enable','on', ...
            'Tag','ColorsCheckBox');
      %
      % Create graph button
      %
      ScatterControls.show = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 55 90 20], ...
         'String','Display', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','ScatterPlotPosteriorDraws showgraph', ...
         'Enable','on', ...
         'Tag','Done');
      %
      %  Done button
      %
      ScatterControls.done = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 20 90 20], ...
         'String','Done', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','ScatterPlotPosteriorDraws done', ...
         'Enable','on', ...
         'UserData','waiting', ...
         'Tag','Done');

      set(ScatterPostDrawsGUI,'UserData',ScatterControls,...
         'HandleVisibility','callback');
   case 'checkparams'
      Param1Val = get(ScatterControls.parameter1popup,'Value');
      Param2Val = get(ScatterControls.parameter2popup,'Value');
      if Param1Val==Param2Val;
         %
         % check which controls should be adjusted
         %
         NumParams = size(get(ScatterControls.parameter1popup,'String'),1);
         if gco==ScatterControls.parameter1popup;
            %
            % parameter 1 has changed
            %
            if Param2Val<NumParams;
               set(ScatterControls.parameter2popup,'Value',Param2Val+1);
            else;
               set(ScatterControls.parameter2popup,'Value',1);
            end;
         else;
            %
            % parameter 2 has changed
            %
            if Param2Val<NumParams;
               set(ScatterControls.parameter1popup,'Value',Param1Val+1);
            else;
               set(ScatterControls.parameter1popup,'Value',1);
            end;
         end;
      end;
   case 'showgraph'
      Param1Value = get(ScatterControls.parameter1popup,'Value');
      Param1String = get(ScatterControls.parameter1popup,'String');
      Param1 = StringTrim(Param1String(Param1Value,:));
      %
      Param2Value = get(ScatterControls.parameter2popup,'Value');
      Param2String = get(ScatterControls.parameter2popup,'String');
      Param2 = StringTrim(Param2String(Param2Value,:));
      %
      % create the figure
      %
      FigHandle =figure('Visible','off', ...
                'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                'Tag',['ScatterPlot-' Param1 '-' Param2], ...
                'Name',['Scatter-Plot of ' Param1 ' and ' Param2]);
      %
      SetFigureProperties(FigHandle,CurrINI);
      %
      % get the data
      %
      x = DSGEModel.ParamSample(:,Param1Value);
      y = DSGEModel.ParamSample(:,Param2Value);
      %
      if MatlabNumber>=7;
         if get(ScatterControls.Colors,'Value')==1;
            ScatHandle = scatter('v6',x,y,4,DSGEModel.LogPost,'filled');
         else;
            ScatHandle = scatter('v6',x,y,4,'filled');
         end;
      else;
         if get(ScatterControls.Colors,'Value')==1;
            ScatHandle = scatter(x,y,4,DSGEModel.LogPost,'filled');
         else;
            ScatHandle = scatter(x,y,4,'filled');
         end;
      end;
      hold('on');
      %
      % add the posterior mode estimate of the first parameter
      %
      xLim = get(gca,'XLim');
      yLim = get(gca,'YLim');
      XParamHandle = plot(DSGEModel.ParamMode(Param1Value)*ones(1,2),yLim,'-','LineWidth',1,'Color',[0 0 0]);
      hold('on');
      YParamHandle = plot(xLim,DSGEModel.ParamMode(Param2Value)*ones(1,2),'-','LineWidth',1,'Color',[0 0 0]);
      hold('off');
      %
      % Show title and labels
      %
      SetAxesFonts(CurrINI,'Title',['Scatter-plot of ' lower(DSGEModel.TypeStr) ' parameters of ' Param1 ' and ' Param2 ' (' num2str(length(x),'%0.0f') ' draws)'],'XLabel',Param1,'Ylabel',Param2);
      if strcmp(get(gca,'YGrid'),'on')==1;
         set(gca,'XGrid','on');
      end;
      %
      SetToolbarImages(FigHandle,CurrINI.images);
      set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
      drawnow;
      pause(0.02);
   case 'done'
      set(ScatterControls.done,'UserData','done');
      delete(ScatterPostDrawsGUI);
      drawnow;
      pause(0.02);
end;
%
if strcmp(selector,'init')==1;
   set(ScatterPostDrawsGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(ScatterControls.done);
   end;
   waitfor(ScatterControls.done,'UserData','done');
end;


%
% end of ScatterPlotPosteriorDraws.m
%
